--[[
	Title: Concommand
	
	Has some functions to make concommands and variables easier.
]]


--[[
	Variable: gFileTimer

	This is used so we're not accessing the file 100 times on server startup

	Revisions:

		v1.2 - Initial.
]]
local gFileTimer


--[[
	Table: gCommands

	Used to support commands, holds access.

	Revisions:

		v1.2 - Initial.
]]
ULib.commands = {}
local gCommands = ULib.commands -- I don't want to go replace all instances of this in the entire file, so let's just alias it.


--[[
	Table: subCommands

	Used to support <BEGIN_SUBCONCOMMAND()> and <ADD_SUBCONCOMMAND()>

	Revisions:

		v1.2 - Added to ULib table, it was *only* local prior to this version.
]]
ULib.subCommands = {}
local gSubCommands = ULib.subCommands -- I don't want to go replace all instances of this in the entire file, so let's just alias it.


--[[
	Function: CONCOMMAND
	
	This is a "beefed up" version of GMod's CONCOMMAND(). Instead of calling the function with just userid and args, it will call the function with userid, args, argv, argc.
	argv is a broken up version of args in a table, obeying quotation marks, argc is the number of values in argv.

	Parameters:

		command - The console command. IE, "ulx_kick".
		fn_call - The function to call when the command's called.
		cmd_help - *(Optional)* A help string for using the command.
		access - *(Optional, defaults to ACCESS_ALL)* The access required for using this command.
		ucl - *(Optional, defaults to ULib.mainUcl)* The ucl to query for access.

	Returns:
	
		True on success, nil and error otherwise.

	Revisions:

		v1 - Uses local function redirects instead of global.
		v1.1 - Added access and ucl params.
		v1.2 - Changed to support dynamic access.
]]
function ULib.CONCOMMAND( command, fn_call, cmd_help, access, ucl )
	if not ULib.checkParams( { command, fn_call }, { "string", "function" } ) then
		return nil, ULib.ERR_ARGS
	end
	
	ucl = ucl or ULib.mainUcl
	gCommands[ command ] = { access=access, help=cmd_help }
	gFileTimer = gFileTimer or ULib.addTimer( 1, 1, ULib.parseAccess )
	
	local fn = function ( userid, args ) -- This function will handle the command callback
		local argv = ULib.splitArgs( args )
		local argc = table.getn( argv )

		if gCommands[ command ] ~= ACCESS_ALL then
			if not ucl:query( userid, gCommands[ command ].access ) then
				ULib.tsay( userid, "You do not have access to this command, " .. _PlayerInfo( userid, "name" ) .. "." )
				-- Print their name to intimidate them :)
				return
			end
		end
		fn_call( userid, args, argv, argc )
	end

	CONCOMMAND( command, fn, cmd_help )
	
	return true
end


--[[
	Function: BEGIN_SUBCONCOMMAND
	
	This function will help you set up subset concommands. For example, instead of using "ulx_kick", "ulx_slay", you may want to use a more conventional format like "ulx kick" and "ulx slay".
	This function helps you acheive that. Call <BEGIN_SUBCONCOMMAND()> first to define the "catch all" function if they specify an unknown parameter of the command by itself ( IE, "ulx" using the examples above ).
	Then, call <ADD_SUBCONCOMMAND()> to add the subconcommands.

	Parameters:
		
		prefix - The console command prefix. IE, "ulx".
		fn_call - The default function to call. This is the fallback if an unknown subcommand is passed, or no additional arguments.
		cmd_help - *(Optional)* A help string for using the command.
		access - *(Optional, defaults to ACCESS_ALL)* The access required to call any command below this, or access the default function.
		ucl - *(Optional, defaults to ULib.mainUcl)* The ucl to query for access.

		
	Returns:
	
		True on success, nil and error otherwise.
		
	Revisions:
	
		v1.1 - Added access and ucl params
		v1.2 - Changed to support dynamic access. Fixed sub-sub-commands
]]
function ULib.BEGIN_SUBCONCOMMAND( prefix, fn_call, cmd_help, access, ucl )
	if not ULib.checkParams( { prefix, fn_call }, { "string", "function" } ) then
		return nil, ULib.ERR_ARGS
	end

	ucl = ucl or ULib.mainUcl
	local prefix_list = ULib.explode( prefix, "%s+" )
	local num = table.getn( prefix_list )
	local t = gSubCommands
	for i=1, num - 1 do -- Get to the table list we need
		t = gSubCommands[ prefix_list[ i ] ]
		if not t then
			return nil, "Recursive subcommands with prefixes that don't exist!"
		end
	end
	t[ prefix_list[ num ] ] = {}
	t = t[ prefix_list[ num ] ] -- This will make it easier to handle

	local f = function ( userid, args, argv, argc ) -- This function will handle the command callback
		if t[ argv[ 1 ] ] then
			local fn = t[ argv[ 1 ] ]

			if fn.access ~= ACCESS_ALL then
				if not fn.ucl:query( userid, fn.access ) then
					ULib.tsay( userid, "You do not have access to this command, " .. _PlayerInfo( userid, "name" ) .. "." )
					-- Print their name to intimidate them :)
					return
				end
			end

			table.remove( argv, 1 )
			if not string.find( args, "%s" ) then
				args = ""
			else
				args = string.gsub( args, "^%S+%s+(.*)$", "%1" )
			end

			fn.fn( userid, args, argv, argc - 1 )
		else
			fn_call( userid, args, argv, argc ) -- Fallback on default function.
		end
	end

	if num == 1 then -- If it's not recursive
		return ULib.CONCOMMAND( prefix, f, cmd_help, access )
	else
		return ULib.ADD_SUBCONCOMMAND( table.concat( prefix_list, " ", 1, num - 1 ), prefix_list[ num ], f, cmd_help, access )
	end
end


--[[
	Function: ADD_SUBCONCOMMAND
	
	See <BEGIN_SUBCONCOMMAND()>.

	Parameters:

		prefix - The console command prefix. IE, "ulx".
		command - The subconcommand. IE, "kick" if you want something like "ulx kick" for the final command.
		fn_call - The function to call when the command's called.
		cmd_help - *(Optional)* A help string for using the command.
		access - *(Optional, defaults to ACCESS_ALL)* The access required for using this command.
		ucl - *(Optional, defaults to ULib.mainUcl)* The ucl to query for access.
		
	Returns:

		True on success, nil and error otherwise.
		
	Revisions:
	
		v1.1 - Added access and ucl params
]]
function ULib.ADD_SUBCONCOMMAND( prefix, command, fn_call, cmd_help, access, ucl )
	if not ULib.checkParams( { prefix, command, fn_call }, { "string", "string", "function" } ) then
		return nil, "Incorrect parameters!"
	end
	
	ucl = ucl or ULib.mainUcl

	local t = ULib.GET_SUBCONCOMMAND( prefix )
	if not t then
		return nil, "Prefix does not exist! You must call BEGIN_SUBCONCOMMAND first."
	end

	t[ command ] = t[ command ] or {}
	t[ command ].fn = fn_call
	t[ command ].access = access
	t[ command ].ucl = ucl
	t[ command ].help = cmd_help

	return true
end


--[[
	Function: GET_SUBCONCOMMAND

	This returns the table associated with a subconcommand. Pass a command with spaces like "ulx slap" to get the table.

	Parameters:

		command - The full subconcommand, including prefixes. IE, "ulx kick".
		
	Returns:

		The table on success, nil and error otherwise.
		
	Revisions:
	
		v1.2 - Initial
]]
function ULib.GET_SUBCONCOMMAND( command )
	local prefix_list = ULib.explode( command, "%s+" )
	local num = table.getn( prefix_list )
	local t = gSubCommands

	for i=1, num do -- Get to the table list we need
		t = t[ prefix_list[ i ] ]
		if not t then
			return nil, "Command doesn't exist!" -- Our command doesn't exist.
		end
	end
	
	return t
end

--[[
	Table: gCvars

	Used to support the cvar functions
]]
local gCvars = {}


--[[
	Function: CONVAR

	Adds a cvar which clients (with permission) can access and change.

	Parameters:

		cvar_name - The cvar name. IE, "ulx_hostname2".
		value - The value to start off at.
		cmd_help - *(Optional)* The help for the command.
		cmd_access - *(Optional, defaults to ACCESS_ALL)* The access for the command.
		ucl - *(Optional, defaults to <ULib.mainUcl>)* The access list to use.

	Returns:

		True on success, nil and error message otherwise.
		
	Revisions:

		v1.2 - Changed to support dynamic access, reverse cmd_help and cmd_access.
]]
function ULib.CONVAR( cvar_name, value, cmd_help, cmd_access, ucl )
	if not ULib.checkParam( cvar_name, "string" ) then return nil, ULib.ERR_ARGS end
	gCvars[ cvar_name ] = value

	local f = function ( userid, args ) -- This function will handle the cvar ( command ) callback
		args = ULib.stripQuotes( args )
		if args == "" then
			ULib.console( userid, "\"" .. cvar_name .. "\" = \"" .. gCvars[ cvar_name ] .. "\"" )
			if cmd_help then
				ULib.console( userid, cmd_help .. "\n  CVAR generated by ULib" )
			else
				ULib.console( userid, "  CVAR generated by ULib" )
			end
		else
			gCvars[ cvar_name ] = args
		end
	end
	
	return ULib.CONCOMMAND( cvar_name, f, cmd_help, cmd_access, ucl )
end


--[[
	Function: ADD_SUBCONVAR

	Adds a sub cvar which clients (with permission) can access and change. See <ADD_SUBCONCOMMAND()>.

	Parameters:

		prefix - The prefix to use, IE: "ulx"
		cvar_name - The cvar name. IE, "hostname2".
		value - The value to start off at.
		cmd_help - *(Optional)* The help for the command.
		cmd_access - *(Optional, defaults to ACCESS_ALL)* The access for the command.
		ucl - *(Optional, defaults to <ULib.mainUcl>)* The access list to use.

	Returns:

		True on success, nil and error message otherwise.
		
	Revisions:

		v1.2 - Changed to support dynamic access, reversed cmd_help and cmd_access.
]]
function ULib.ADD_SUBCONVAR( prefix, cvar_name, value, cmd_help, cmd_access, ucl )
	if not ULib.checkParams( { prefix, cvar_name }, { "string", "string" } ) then return nil, ULib.ERR_ARGS end
	gCvars[ prefix .. " " .. cvar_name ] = value

	local f = function ( userid, args ) -- This function will handle the cvar ( command ) callback
		args = ULib.stripQuotes( args )
		if args == "" then
			if cmd_help then
				ULib.console( userid, "\"" .. cvar_name .. "\" = \"" .. gCvars[ prefix .. " " .. cvar_name ] .. "\"" )
				ULib.console( userid, cmd_help .. "\n  CVAR generated by ULib" )
				return
			else
				ULib.console( userid, "\"" .. cvar_name .. "\" = \"" .. gCvars[ prefix .. " " .. cvar_name ] .. "\"" )
				ULib.console( userid, "  CVAR generated by ULib" )
				return
			end
		else
			gCvars[ prefix .. " " .. cvar_name ] = args
		end
	end

	return ULib.ADD_SUBCONCOMMAND( prefix, cvar_name, f, cmd_help, cmd_access, ucl )
end


--[[
	Function: getCvar

	Gets the string value from a cvar

	Parameters:

		cvar_name - The name of the cvar to retrieve
]]
function ULib.getCvar( cvar_name )
	return gCvars[ cvar_name ]
end

--[[
	Function: parseAccess
	
	*For internal use only, do not call directly!*

	Revisions:

		v1.2 - Initial
]]
function ULib.parseAccess()
	gFileTimer = nil -- So it can be called again later

	local data = {}
	if _file.Exists( ULib.ACCESS_FILE ) then
		data = ULib.parseKeyValues( ULib.stripComments( _file.Read( ULib.ACCESS_FILE ), "//" ) )
		if data == nil then data = {} end
	end
	
	local reverse_table = {}
	for access, commands in pairs( data ) do
		for index, command in ipairs( commands ) do
			if reverse_table[ command ] then ULib.print( "\n\n\n[ULIB] ERROR! Found a duplicate command \"" .. command .. "\" in command_access.ini!\n\n\n" ) end
			reverse_table[ command ] = access  -- We'll use this to cycle through and see if a commands missing later.

			if gCommands[ command ] then -- If it even exists as a command. (IE, was this script removed?)
				gCommands[ command ].access = ULib.access[ access ] or access -- Set the access right, picks up defines.
				if access == "ACCESS_ALL" then
					gCommands[ command ].access = ACCESS_ALL -- Because this defauls to nil, it requires special handling.
				end

			elseif string.find( command, " " ) then -- If it's got spaces, it must be a subconcommand
				local t = ULib.GET_SUBCONCOMMAND( command )
				if t then -- If it exists
					t.access = ULib.access[ access ] or access -- Set the access right, picks up defines.
					if access == "ACCESS_ALL" then
						t.access = ACCESS_ALL -- Because this defauls to nil, it requires special handling.
					end
				end
			end

			if ULib.reverseAccess[ access ] then -- If we can change it to a define, do it!
				local newAccess = ULib.reverseAccess[ access ]
				table.remove( data[ access ], index ) -- Delete it
				index = index - 1 -- So we don't skip a value, since we just deleted the current one.

				if not ULib.isInTable( data[ newAccess ], command ) then -- If it's not already in here...
					data[ ULib.reverseAccess[ access ] ] = data[ newAccess ] or {} -- Make sure the new table exists
					table.insert( data[ newAccess ], command )
				end
			end
		end
		if table.getn( data[ access ] ) == 0 then data[ access ] = nil end -- If the table's empty, delete it.
	end

	---------------------------------------------------------------------------------------
	--We will now check for commands that are NOT in the table, and add them to the file.--
	---------------------------------------------------------------------------------------

	for command, v in pairs( gCommands ) do -- Do regular concommands first
		if not reverse_table[ command ] then -- If it's not in the file, we need to put it there!
			local access_string = v.access
			if v.access == ACCESS_ALL then
				access_string = "ACCESS_ALL"
			elseif ULib.reverseAccess[ v.access ] then
				access_string = ULib.reverseAccess[ v.access ]
			else
				access_string = v.access
			end
			data[ access_string ] = data[ access_string ] or {}
			table.insert( data[ access_string ], command )
		end
	end

	local function parseSubCommands( t, prefix ) -- Do subcommands now
		local newprefix = prefix
		if newprefix ~= "" then newprefix = newprefix .. " " end -- So we don't get " ulx kick"

		for command, v in pairs( t ) do
			if v.ucl and v.fn then -- If it's a subconcommand
				if not reverse_table[ newprefix .. command ] then -- and if it's not in the file, we need to put it there!
					local access_string = v.access
					if v.access == ACCESS_ALL then
						access_string = "ACCESS_ALL"
					elseif ULib.reverseAccess[ v.access ] then
						access_string = ULib.reverseAccess[ v.access ]
					else
						access_string = v.access
					end
					data[ access_string ] = data[ access_string ] or {}
					table.insert( data[ access_string ], newprefix .. command )
				end
			else
				parseSubCommands( t[ command ], newprefix .. command )
			end
		end
	end
	parseSubCommands( gSubCommands, "" ) -- Start the recursion.

	comment = [[//This file lets you easily change the access required to use commands. If there's not a table set up for an access you want to use,
//feel free to make it. You can either use defines (ACCESS_ALL), or the letters ("b"). If you use letters, they'll be converted to defines
//on the next load (remember letters are case-sensitive). DO NOT MESS WITH COMMANDS UNDER ACCESS_NONE UNLESS YOU KNOW WHAT YOU'RE DOING!
]]

	_file.Write( ULib.ACCESS_FILE, comment .. ULib.makeKeyValues( data ) )
	--ULib.print( "finished" )
end