--[[
	Title: Miscellaneous

	Some utility functions. Unlike the functions in util.lua, this file only holds non-HL2 specific functions.
]]


--[[
	Function: serialize

	Serializes a variable. It basically converts a variable into a runnable code string. It works correctly with inline tables.

	Parameters:

		v - The variable you wish to serialize
		space - *(Optional, defaults to <SPACE_METHOD>)* The starting space method to use, used to show structure ( IE: inline tables would have extra spacing ).
		name - *(Optional)* If you're saving a table, you should specify the name to help facilitate recursive checks.
		completed - *(Only for internal use)* This parameter assures that we won't loop infinitely by keeping track of what tables we've done.

	Returns:

		The string of the serialized variable

	Example:

        	*(input)*
		: _Msg( serialize( { "hi\nhowdy!", "doh!", { { hi = "yo" }, yay = "test" } } ) )
		*(output)*
		:{
		:        1 = "hi\nhowdy!",
		:        2 = "doh!",
		:        3 = {
		:                1 = {
		:                        hi = "yo",
		:                },
		:                yay = "test",
		:        },
		:}

	Example 2:

		*(input)*
		: _Msg( serialize( 47 ) )
		*(output)*
		: 47
		
	Revisions:
	
		v1 - Can now handle funky key strings.
]]
function ULib.serialize( v, space, name, completed )
	completed = completed or {}

	local s = ""
	space = space or ULib.SPACE_METHOD
	if type( v ) == "string" then
		s = string.format( "%q", v )
	elseif type( v ) == "table" then
		if completed[ v ] then
			if type( completed[ v ] ) == "string" then
				return completed[ v ]
			else
				return "{}" .. ULib.NEWLINE
			end
		end
		completed[ v ] = ( name or true )

		s = "{" .. ULib.NEWLINE
		for key in pairs( v ) do
			if type( key ) == "string" then
				s = s .. space .. "[" .. string.format( "%q", key ) .. "]" .. " = "
			else
				s = s .. space .. key .. " = "
			end
			s = s .. ULib.serialize( v[ key ], space .. ULib.SPACE_METHOD, name .. "." .. key, completed )
			s = s .. "," .. ULib.NEWLINE
		end
		s = s .. string.sub( space, string.len( ULib.SPACE_METHOD ) + 1 ) .. "}"
	elseif type( v ) == "number" or type( v ) == "boolean" or type( v ) == "nil" then
		s = tostring( v )
	else
		ULib.debug( "Cannot serialize a " .. type( v ), ULib.err )
	end
	return s
end


--[[
	Function: explode

	Split a string by a separator.

	Parameters:

		separator - The separator string.
		str - A string.
		limit - *(Optional)* Max number of elements in the table

	Returns:

		A table of str split by separator, nil and error message otherwise.

	Revisions:

		v1- Added limit argument
]]
function ULib.explode( str, separator, limit )
        if not ULib.checkParams( { str, separator }, { "string", "string" } ) then return nil, ULib.ERR_ARGS end

	local t = {}
	local curpos = 1
	while true do -- We have a break in the loop
		local newpos, endpos = string.find( str, separator, curpos ) -- find the next separator in the string
		if newpos ~= nil then -- if found then..
			table.insert( t, string.sub( str, curpos, newpos - 1 ) ) -- Save it in our table.
			curpos = endpos + 1 -- save just after where we found it for searching next time.
		else
			if limit and table.getn( t ) > limit then
				return t -- Reached limit
			end
			table.insert( t, string.sub( str, curpos ) ) -- Save what's left in our array.
			break
		end
	end

	return t
end


--[[
	Function: implode

	Takes characters in a table and rebuild them into a string.
	Note that this simply re-routes the call to table.concat()

	Parameters:

		t - A table containing the characters in the string to rebuild, in respective order.
		sep - A string to place between each value in the table.
		i - *(Optional, defaults to 1)* Where to start within the table.
		j - *(Optional, defaults to the length of the table)* Where to end.

	Returns:

		A a string with seperators between each value in t. If i > j it will return an empty string
]]
function ULib.implode ( t, sep, i, j )
	return table.concat( t, sep, i, j )
end


--[[
	Function: trim

	Trims leading and tailing whitespace from a string

	Parameters:

		s - The string to trim

	Returns:

		The trimmed string
]]
function ULib.trim( s )
	return string.gsub( s, "^%s*(.-)%s*$", "%1" )
end


--[[
	Function: stripQuotes

	Trims leading and tailing quotes from a string

	Parameters:

		s - The string to strip

	Returns:

		The stripped string
]]
function ULib.stripQuotes( s )
	return string.gsub( s, "^%s*[\"]*(.-)[\"]*%s*$", "%1" )
end


--[[
	Function: checkParam

	Checks a parameter against a type.

	Parameters:

		value - The value to check
		t - The string type the value should be. If you specify "string", it will make sure it's not an empty string.

	Returns:

		True if the value/type match, false otherwise.
]]
function ULib.checkParam( value, t )
	if type( t ) ~= "string" then return false end

        local vt = type( value )
	if vt == t then
		if vt == "string" and value == "" then return false end
		return true
	end
	
	return false
end


--[[
	Function: checkParams

	Checks parameters against types.

	Parameters:

		values - A table of the values to check. The two tables ( values and ts ) *must* be the same length.
		ts - The table containing the types the values should be. If you specify "string", it will make sure it's not an empty string.

	Returns:

		True if the values and types match, false otherwise.
]]
function ULib.checkParams( values, ts )
	if type( ts ) ~= "table" or type( values ) ~= "table" then return false end
	if table.getn( values ) ~= table.getn( ts ) then return false end

	for i=1, table.getn( values ) do
		if not ULib.checkParam( values[ i ], ts[ i ] ) then return false end
	end

	return true
end


--[[
	Function: stripComments

	Strips comments from a string

	Parameters:

		str - The string to stip comments from
		comment - The comment string. If it's found, whatever comes after it on that line is ignored. ( IE: "//" )
		blockcommentbeg - *(Optional)* The block comment begin string. ( IE: "/*" )
		blockcommentend - *(Optional, must be specified if above parameter is)* The block comment end string. ( IE: "*/" )

	Returns:

		The string with the comments stripped, nil and error otherwise.
]]
function ULib.stripComments( str, comment, blockcommentbeg, blockcommentend )
	if not ULib.checkParams( { str, comment }, { "string", "string" } ) then return nil, ULib.ERR_ARGS end
	if blockcommentbeg then if not ULib.checkParams( { blockcommentbeg, blockcommentend }, { "string", "string" } ) then return nil, ULib.ERR_ARGS end end

	if blockcommentbeg and string.sub( blockcommentbeg, 1, string.len( comment ) ) == comment then -- If the first of the block comment is the linecomment ( IE: --[[ and -- ).
		string.gsub( str, ULib.makePatternSafe( comment ) .. "[%S ]*", function ( match )
			if string.sub( match, 1, string.len( blockcommentbeg ) ) == blockcommentbeg then
				return -- No substitution, this is a block comment.
			end
			str = string.gsub( str, match, "", 1 )
		end )

		str = string.gsub( str, ULib.makePatternSafe( blockcommentbeg ) .. ".-" .. ULib.makePatternSafe( blockcommentend ), "" )
	else -- Doesn't need special processing.
		str = string.gsub( str, ULib.makePatternSafe( comment ) .. "[%S ]*", "" )
		if blockcommentbeg and blockcommentend then
			str = string.gsub( str, ULib.makePatternSafe( blockcommentbeg ) .. ".-" .. ULib.makePatternSafe( blockcommentend ), "" )
		end
	end
	
	return str
end


--[[
	Function: makePatternSafe

	Makes a string safe for pattern usage, like in string.gsub(). Basically replaces all keywords with % and keyword.

	Parameters:

		str - The string to make pattern safe

	Returns:

		The pattern safe string
]]
function ULib.makePatternSafe( str )
         return string.gsub( str, "([%(%)%.%%%+%-%*%?%[%]%^%$])", "%%%1" )
end


--[[
	Function: splitArgs

	This is similar to explode( str, "%s" ) except that it will also obey quotation marks.

	Parameters:

		str - The string to split from

	Returns:

		A table containing the individual arguments
		
	Example:
	
		:ULib.splitArgs( "This is a \"Cool sentence to\" make \"split up\"" )

		returns...

		{ "This", "is", "a", "Cool sentence to", "make", "split up" }
		
	Revisions:
	
		v1 - Can now handle empty strings.
]]
function ULib.splitArgs( str )
	if type( str ) ~= "string" then return nil, ULib.ERR_ARGS end
	str = ULib.trim( str )
	if str == "" then return {} end
	local quotes = {}
	local t = {}
	local marker = "^*#" -- This is used as a placeholder
         
	string.gsub( str, "%b\"\"", function ( match ) -- We're finding the quote groups first.
		local s = ULib.stripQuotes( match )
		table.insert( quotes, s )

		str = string.gsub( str, ULib.makePatternSafe( match ), marker .. table.getn( quotes ), 1 ) -- We aren't sure where in the table this should be placed, so let's leave a marker.
	end )

	t = ULib.explode( str, "%s+" )

	for i, v in ipairs( t ) do
		if string.sub( v, 1, 3 ) == marker then -- Now we can read our marker into the correct place in the table.
			local num = tonumber( string.sub( v, 4 ) )
			t[ i ] = quotes[ num ]
		end
	end

	return t
end


--[[
	Function: isStringTrue
	
	Returns whether a string should be evaluated as true or not. Supports "1"-"0", "true"-"false", and "yes"-"no".

	Parameters:

		str - The string to evaluate.

	Returns:

		True or false, nil if it's not recognized.

	Revisions:
	
		v1.1 - Initial
]]
function ULib.isStringTrue( str )
	if not ULib.checkParam( str, "string" ) then return nil, ULib.ERR_ARGS end
	
	if str == "true" or str == "1" or str == "yes" then return true end
	if str == "false" or str == "0" or str == "no" then return false end

	return nil
end


--[[
	Function: sToHMS

	Takes a parameter of seconds and return hours, minutes, and seconds.
	
	Parameters:
	
		secs - The number of seconds.

	Returns:

		Hours, minutes, seconds.

	Revisions:

		v1.2 - Initial
]]
function ULib.sToHMS( secs )
	local s = math.floor( secs )
	local m = math.floor(s/60)
	local h = math.floor(m/60)

	s = math.mod(s,60)
	m = math.mod(m,60)

	return h, m, s
end


--[[
	Function: getTimeStamp
	
	Get a timestamp from seconds
	
	Parameters:
	
		sec - The number of seconds you want the timestamp for
		noHours - *(Optional, defaults to false)* If this option is false, it will return in "MM:SS" format, dropping hours.

	Returns:

		The timestamp in format "HH:MM:SS" since start of map

	Revisions:

		v1.2 - Initial
]]
function ULib.getTimeStamp( sec, noHours )
	local s = math.floor( sec )
	local m = math.floor(s/60)
	local h = math.floor(m/60)

	s = math.mod(s,60)
	
	if not noHours then
		m = math.mod(m,60)
	end

	s = string.format( "%02i", s )
	m = string.format( "%02i", m )
	h = string.format( "%02i", h )

	if not noHours then return h .. ":" .. m .. ":" .. s
	else return m .. ":" .. s end
end