--[[
	Title: Player

	Has useful player-related functions.
]]


--[[
	Table: gSlapSounds

	These are the sounds used for slaps.
]]
gSlapSounds = {
	"physics/body/body_medium_impact_hard1.wav",
	"physics/body/body_medium_impact_hard2.wav",
	"physics/body/body_medium_impact_hard3.wav",
	"physics/body/body_medium_impact_hard5.wav",
	"physics/body/body_medium_impact_hard6.wav",
	"physics/body/body_medium_impact_soft5.wav",
	"physics/body/body_medium_impact_soft6.wav",
	"physics/body/body_medium_impact_soft7.wav",
}


--[[
	Function: slap

	Slaps a targetid, can be a user or any entity.

	Parameters:

		target - Target id.
		damage - *(Optional, defaults to 0)* The amount of damage to inflict on the entity.
		power - *(Optional, defaults to 30)* The power of the slap.
		nosound - *(Optional, defaults to false)* If true, no sound will be played.
]]
function ULib.slap( target, damage, power, nosound )
	if _EntGetMoveType( target ) == MOVETYPE_OBSERVER then return end -- Nothing we can do.

	damage = damage or 0
	power = power or 30
	
	if ULib.isInVehicle( target ) then ULib.forceOutOfVehicle( target ) end

	if not nosound then -- Play a slap sound
		local sound_num = math.random( table.getn( gSlapSounds ) ) -- Choose at random
		ULib.play3DSound( gSlapSounds[ sound_num ], _EntGetPos( target ), 1.5 )
	end

	local direction = vector3( math.random( 20 )-10, math.random( 20 )-10, math.random( 20 )-5 ) -- Make it random, slightly biased to go up.
	ULib.applyAccel( target, power, direction )

	local newHp = _PlayerInfo( target, "health" ) - damage
	if newHp <= 0 then
		_PlayerKill( target )
		return
	end
	_PlayerSetHealth( target, newHp )
end


--[[
	Function: forceOutOfVehicle

	Forces a player out of a vehicle.

	Parameters:

		userid - The player to force out of a vehicle.
]]
function ULib.forceOutOfVehicle( userid )
	if not ULib.isInVehicle( userid ) then
		return nil, "Userid is not in a vehicle"
	end

	local vehicle = _EntGetParent( userid )
	local playerPos = _EntGetPos( userid )
	local playerPosNew = vecAdd( playerPos, vecMul( _EntGetRightVector( vehicle ), -80 ) )

	_PlayerGod( userid, false ) -- Let him be killed
	_PlayerKill( userid )
	_PlayerRespawn( userid )
	_EntSetPos( userid, playerPosNew )
end


--[[
	Function: isInVehicle

	Checks if the entity (player) is in a vehicle

	Parameters:

		userid - The userid to check
	
	Returns:
	
		True if they're in a vehicle, false otherwise.
]]
function ULib.isInVehicle( userid )
	if string.find( _EntGetType( _EntGetParent( userid ) ), "vehicle" ) then
		return true
	else
		return false
	end
end


--[[
	Table: gSteamIdToIp

	This will hold the IPs of connected steamids
]]
local gSteamIdToIp = {}


--[[
	Table: gNameToIp
	
	This holds usernames for <relateUserIp()>
]]
local gNameToIp = {}


--[[
	Function: collectUserIp

	*THIS IS FOR INTERNAL USE*. This is what gives <gSteamIdToIp> it's values.
]]
function ULib.collectUserIp( name, ip, steamid )
	local beg = string.find( ip, ":" )
	if not beg then gNameToIp[ name ] = "none" return end
	ip = string.sub( ip, 1, beg - 1 )
	gNameToIp[ name ] = ip
end
HookEvent( "eventPlayerConnect", ULib.collectUserIp )


--[[
	Function: relateUserIp

	*THIS IS FOR INTERNAL USE*. This is what gives <gSteamIdToIp> it's values.
	We have to use two functions, one to catch the ip, the other to catch the steamid.
]]
function ULib.relateUserIp( name, userid, steamid )
	gSteamIdToIp[ steamid ] = gNameToIp[ name ]
	gNameToIp[ name ] = nil
end
HookEvent( "eventPlayerActive", ULib.relateUserIp )


--[[
	Function: getIp

	Parameters:

		userid - The user's id
		
	Returns:
	
		The user's IP
]]
function ULib.getIp( userid )
	return gSteamIdToIp[ _PlayerInfo( userid, "networkid" ) ]
end


--[[
	Function: getUsers

	Finds users matching an identifier.

	Parameters:

		identifier - A string of the user you'd like to target. IE, a partial player name. Also recognizes a userid string.
		ignore_immunity - *(Optional, defaults to false)* If true, users with immunity will be returned.
		enable_keywords - *(Optional, defaults to false)* If true, keywords "<ALL>", "<BLUE>", "<YELLOW>", "<GREEN>",
			and "<RED>" will be activated. They will return what you think they would (All players or teams).
		ucl - *(Optional, defaults to ULib.mainUcl)* The access list to use.
		userid - *(Optional)* Userid needing getUsers, this allows immune users to target themselves.

	Returns:

		A table of userids (false and message if none found), nil and error message otherwise.
		
	Revisions:
	
		v1.1 - Fixed immunity in keywords, added userid parameter.
		v1.2 - FIX, was not returning an empty table on no users found (as docs previously stated), now returns false and an error.
]]
function ULib.getUsers( identifier, ignore_immunity, enable_keywords, ucl, userid )
	if not ULib.checkParam( identifier, "string" ) then
		return nil, ULib.ERR_ARGS
	end

	ucl = ucl or ULib.mainUcl
	userid = userid or 0

	local maxplayers = _MaxPlayers()
	local result = {}
	
	if enable_keywords then
		if string.upper( identifier ) == "<ALL>" then
			for i=1, maxplayers do
				if _PlayerInfo( i, "connected" ) and ( userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) ) then
					table.insert( result, i )
				end
			end
			return result
		elseif string.upper( identifier ) == "<BLUE>" then
			for i=1, maxplayers do
				if _PlayerInfo( i, "connected" ) and _PlayerInfo( i, "team" ) == TEAM_BLUE and ( userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) ) then
					table.insert( result, i )
				end
			end
			return result
		elseif string.upper( identifier ) == "<YELLOW>" then
			for i=1, maxplayers do
				if _PlayerInfo( i, "connected" ) and _PlayerInfo( i, "team" ) == TEAM_YELLOW and ( userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) ) then
					table.insert( result, i )
				end
			end
			return result
		elseif string.upper( identifier ) == "<GREEN>" then
			for i=1, maxplayers do
				if _PlayerInfo( i, "connected" ) and _PlayerInfo( i, "team" ) == TEAM_GREEN and ( userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) ) then
					table.insert( result, i )
				end
			end
			return result
		elseif string.upper( identifier ) == "<RED>" then
			for i=1, maxplayers do
				if _PlayerInfo( i, "connected" ) and _PlayerInfo( i, "team" ) == TEAM_RED and ( userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) ) then
					table.insert( result, i )
				end
			end
			return result
		end
	end

	local num = tonumber( identifier )
	if num and num > 0 and num <= maxplayers and _PlayerInfo( num, "connected" ) then -- Guess they mean a person's userid
		if userid == num or ignore_immunity or not ucl:query( num, ACCESS_IMMUNITY ) then -- If we're ignoring immunity or if they don't have immunity.
			return { num } -- And we're done!
		else
			return nil, "Passed a userid, but the userid had immunity"
		end
	end
	
	local user = string.lower( identifier ) -- We want to make this a case insensitive search

	for i=1, maxplayers do
		if _PlayerInfo( i, "connected" ) == true then
			temp = string.find( string.lower( _PlayerInfo( i, "name" ) ), user, 1, true ) -- No special characters.
			if temp then
				if userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) then -- If we're ignoring immunity or if they don't have immunity.
					table.insert( result, i )
				end
			end
		end
	end

	if table.getn( result ) < 1 then
		return false, "No target found or target has immunity!"
	end

	return result
end


--[[
	Function: getUser

	Finds a user matching an identifier.

	Parameters:

		identifier - A string of the user you'd like to target. IE, a partial player name. Also recognizes a userid string.
		ignore_immunity - *(Optional, defaults to false)* If true, users with immunity will be returned.
		ucl - *(Optional, defaults to ULib.mainUcl)* The access list to use.
		userid - *(Optional)* Userid needing getUser, this allows immune users to target themselves.

	Returns:

		The resulting userid target, false and message if no user found. In all other cases, nil and a message.
		
	Revisions:
	
		v1.1 - Added userid parameter.
		v1.2 - Now returns false on no user found.
]]
function ULib.getUser( identifier, ignore_immunity, ucl, userid )
	if not ULib.checkParam( identifier, "string" ) then
		return nil, ULib.ERR_ARGS
	end

	ucl = ucl or ULib.mainUcl
	userid = userid or 0

	local result

	local num = tonumber( identifier )
	if num and num > 0 and num <= _MaxPlayers() and _PlayerInfo( num, "connected" ) then -- Guess they mean a person's userid		
		if userid == num or ignore_immunity or not ucl:query( num, ACCESS_IMMUNITY ) then -- If we're ignoring immunity or if they don't have immunity.
			return num -- And we're done!
		else
			return nil, "Passed a userid, but the userid had immunity"
		end
	end
	
	local user = string.lower( identifier ) -- We want to make this a case insensitive search

	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) == true then
			temp = string.find( string.lower( _PlayerInfo( i, "name" ) ), user, 1, true ) -- No special characters.
			if temp then  -- We have a match
				if result then
					return nil, "Found multiple targets! Please choose a better string for the target."
				end

				if userid == i or ignore_immunity or not ucl:query( i, ACCESS_IMMUNITY ) then -- If we're ignoring immunity or if they don't have immunity.
					result = i
				else
					return nil, "That user has immunity!"
				end
			end
		end
	end

	if result then return result end
	
	return false, "No target found!"
end


--[[
	Function: kick

	Kicks a user.

	Parameters:

		userid - The userid to kick.
		reason - *(Optional)* The reason to give for kicking. *Note, providing a reason makes this a little unstable. If you want to kick bots or people on a LAN server, do NOT provide a reason.*
]]
function ULib.kick( userid, reason )
	if reason and type( reason ) == "string" then
		_ServerCommand( "kickid " .. _PlayerInfo( userid, "networkid" ) .. " \"" .. reason .. "\"\n" )
	else
		_ServerCommand( "kick " .. _PlayerInfo( userid, "name" ) .. "\n" )
	end
end


--[[
	Function: ban

	Bans a user.

	Parameters:

		userid - The userid to ban.
		time - *(Optional)* The time in minutes to ban the person for, leave nil or 0 for permaban.
		
	Revisions:
	
		v1 - Fixed
		v1.2 - Fixed "bans mysteriously disappear" bug. (I blame VALVe!)
]]
function ULib.ban( userid, time )
	if not time or type( time ) ~= "number" then
		time = 0
	end

	-- Load our currently banned users so we don't overwrite them
	if _file.Exists( "cfg/banned_user.cfg" ) then
		ULib.execFile( "cfg/banned_user.cfg" )
	end

	_ServerCommand( "banid " .. time .. " " .. _PlayerInfo( userid, "networkid" ) .. ";writeid\n" )
end


--[[
	Function: kickban

	Kicks and bans a user.

	Parameters:

		userid - The userid to kickban.
		time - *(Optional)* The time in minutes to ban the person for, leave nil or 0 for permaban.

	Revisions:

		v1 - Fixed
		v1.2 - Fixed "bans mysteriously disappear" bug. (I blame VALVe!)
]]
function ULib.kickban( userid, time )
	if not time or type( time ) ~= "number" then
		time = 0
	end
	
	-- Load our currently banned users so we don't overwrite them
	if _file.Exists( "cfg/banned_user.cfg" ) then
		ULib.execFile( "cfg/banned_user.cfg" )
	end

	_ServerCommand( "banid " .. time .. " " .. _PlayerInfo( userid, "networkid" ) .. " kick;writeid\n" )
end


--[[
	Function: getSteamLogin

	Retrieves a user's login name.

	Parameters:

		userid - The entity index location of the user.

	Returns:
	
		A string of the user's Steam login name. This is the username they use when logging into Steam. Returns "BOT" for bots.
		
	Revisions:
	
		v1.1 - Added bot handling
]]
function ULib.getSteamLogin( userid )
	local login = _GetClientConVar_String( userid, "stmlgn" )
	if login == "" and _PlayerInfo( 2, "networkid" ) == "BOT" then
		login = "BOT"
	end
	return login
end

--- Below are helper functions for frozen players
local frozen = {}

local function freezeSpawnHook( userid )
	if frozen[ userid ] then
		_PlayerFreeze( userid, true )
		_EntSetPos( userid, frozen[ userid ].pos )
		_EntSetAng( userid, frozen[ userid ].ang )
		_EntSetMoveType( userid, MOVETYPE_NONE ) -- This makes them not move with explosions and such
	end
end

local function freezeDisconnectHook( name, userid )
	if frozen[ userid ] then
		frozen[ frozen[ userid ].login ] = frozen[ userid ]
		frozen[ userid ] = nil
	end
end

local function freezeInitialSpawnHook( userid )
	if frozen[ ULib.getSteamLogin( userid ) ] then
		local info = frozen[ ULib.getSteamLogin( userid ) ]
		frozen[ ULib.getSteamLogin( userid ) ] = nil
		_EntSetPos( userid, info.pos )
		_EntSetAng( userid, info.ang )
		ULib.freezePlayer( userid )
		_EntSetMoveType( userid, MOVETYPE_NONE ) -- This makes them not move with explosions and such
	end
end

local function freezePropHook( userid )
	if frozen[ userid ] then
		return false
	end
	
	return frozen.eventPlayerSpawnProp
end

local function freezeRagdollHook( userid )
	if frozen[ userid ] then
		return false
	end
	
	return frozen.eventPlayerSpawnRagdoll
end

--[[
	Function: freezePlayer

	Freezes a player. Unlike _PlayerFreeze, this is a *complete* freeze. They will not be able to spawn props, kill to get free, reconnect to get free, etc.

	Parameters:

		userid - The user to freeze
]]
function ULib.freezePlayer( userid )
	frozen[ userid ] = { pos=_EntGetPos( userid ), ang=_PlayerGetShootAng( userid ), login=ULib.getSteamLogin( userid ) }
	_PlayerFreeze( userid, true )
	_EntSetMoveType( userid, MOVETYPE_NONE ) -- This makes them not move with explosions and such

	if not frozen.hookid then
		frozen.hookid = HookEvent( "eventPlayerSpawn", freezeSpawnHook )
		frozen.hookid2 = HookEvent( "eventPlayerInitialSpawn", freezeInitialSpawnHook )
		frozen.hookid3 = HookEvent( "eventPlayerDisconnect", freezeDisconnectHook )
		
		frozen.eventPlayerSpawnProp = eventPlayerSpawnProp
		frozen.eventPlayerSpawnRagdoll = eventPlayerSpawnRagdoll
		eventPlayerSpawnProp = freezePropHook
		eventPlayerSpawnRagdoll = freezeRagdollHook
	end
end


--[[
	Function: unfreezePlayer

	Unfreezes a player.

	Parameters:

		userid - The user to unfreeze
]]
function ULib.unfreezePlayer( userid )
	frozen[ userid ] = nil
	_PlayerFreeze( userid, false )
	_EntSetMoveType( userid, MOVETYPE_WALK ) -- Reset their movetype

	local delete = true
	
	for k, v in pairs( frozen ) do
		if k ~= "eventPlayerSpawnProp" and k~= "eventPlayerSpawnRagdoll" and string.sub( k, 1, 6 ) ~= "hookid" then
			delete = false
			break
		end
	end

	if delete then
		UnHookEvent( frozen.hookid )
		UnHookEvent( frozen.hookid2 )
		UnHookEvent( frozen.hookid3 )
		frozen.hookid = nil
		frozen.hookid2 = nil
		frozen.hookid3 = nil

		eventPlayerSpawnProp = frozen.eventPlayerSpawnProp -- Return functions to their natural order
		eventPlayerSpawnRagdoll = frozen.eventPlayerSpawnRagdoll
	end
end


--[[
	Function: playerString

	Returns "username(user_steamid)" for a userid. Useful for debug messages.

	Parameters:

		userid - The user
		
	Returns:
	
		The user's name .. "(" .. the user's steamid .. ")"
]]
function ULib.playerString( userid )
	return _PlayerInfo( userid, "name" ) .. "(" .. _PlayerInfo( userid, "networkid" ) .. ")"
end


--[[
	Function: callcmd

	Calls a command as if the specified userid were calling it.

	Parameters:

		userid - The user that is "calling" the command.
		str - The string, includes command as well as args.

	Revisions:

		v1.1 - Initial
]]
function ULib.callCmd( userid, str )
	local words = ULib.splitArgs( str )
	if gConsoleCommands[ words[ 1 ] ] then
		local args = string.gsub( str, "^%s*%S+%s*(.*)$", "%1" )
		gConsoleCommands[ words[ 1 ] ]( userid, args )
	end
end


--[[
	Function: totalPlayers

	Returns how many players are currently connected to the server.
	
	Returns:
	
		Total players in server.

	Revisions:

		v1.2 - Initial
]]
function ULib.totalPlayers()
	local count = 0
	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) then
			count = count + 1
		end
	end
	
	return count
end