--[[
	Title: Menu

	ULib's very own menu generator.
	
	The format for the menu callbacks are ( userid, optionid, time, option_text )
	userid - I'm sure you can guess. ;)
	optionid - The ID of the option they chose, you give options ids when you make them.
	time - The time it took for them to choose this option.
	option_text - The text that was on this option.
]]



--[[
	Class: Menu

	This handles ULib's menu.
]]
ULib.Menu = {}


--[[
	Function: new
	
	Creates a new menu.

	Parameters:

		title - The title of the menu.
		callback - The function to callback.
		skin - *(Optional, defaults to ULib.default_skin)* The skin to use for the menu.
		t - *(Optional)* A table to use to start out with.

	Returns:

		A new instantiation of the Menu class, nil and error message otherwise.
]]
function ULib.Menu:new( title, callback, skin, t )
	skin = skin or ULib.default_skin
	t = t or {}   -- Create object if user does not provide one

	if not ULib.checkParams( { title, callback, skin, t }, { "string", "function", "table", "table" } ) then return nil, ULib.ERR_ARGS end

	setmetatable( t, self ) -- If you're unfamiliar with metatables read <http://www.lua.org/pil/13.html>
	self.__index = self

	t.title = title
	t.callback = callback
	t.skin = skin
	t.skin.prev = t.skin.prev or "Previous Page"
	t.skin.next = t.skin.next or "Next Page"
	t.skin.exit = t.skin.exit or "Exit"
	t.skin.bind_options = t.skin.bind_options
	if t.skin.bind_options == nil then
		t.skin.bind_options = true
	end
	t.options = {}

	return t
end


--[[
	Function: addOption
	
	Adds a new option to the menu.

	Parameters:

		text - The text to display for the option. IE: Kick Bob.
		id - *(Optional)* The ID to use for the option. This is what is sent to the callback when it's chosen. If you don't specify an ID, the option # is used.
		access - *(Optional)* Only shows the option to the user if they have access. Note that access is ignored on menus sent to all players!
		callback - *(Optional)* If a callback is specified, this callback will be used for this option instead of the one specified in new.
		num - *(Optional)* The option number this should be. IE, you can force something to be first. If no number is specified, it's inserted as the last option.

	Returns:

		True on success, nil and error message otherwise.
]]
function ULib.Menu:addOption( text, id, access, callback, num )
	if not num then
		num = table.getn( self.options ) + 1
	end

	if not id then
		id = num
	end

	if not ULib.checkParams( { text, num }, { "string", "number" } ) or (callback and not ULib.checkParam( callback, "function" )) then 
		return ULib.ERR_ARGS 
	end

	table.insert( self.options, num, { text=text, id=id, access=access, callback=callback } )

	return true
end


--[[
	Function: addRconOption
	
	Adds a new rcon option to the menu. If this option is chosen, an rcon command will execute.

	Parameters:

		text - The text to display for the option. IE: Kick Bob
		command - The command to execute at the server console. IE: "kick bob"
		access - *(Optional)* Only shows the option to the user if they have access.
		num - *(Optional)* The option number this should be. IE, you can force something to be first. If no number is specified, it's inserted as the last option.

	Returns:

		True on success, nil and error message otherwise.
]]
function ULib.Menu:addRconOption( text, command, access, num )
	if not num then
		num = table.getn( self.options ) + 1
	end

	if not ULib.checkParams( { text, command, num }, { "string", "string", "number" } ) then return ULib.ERR_ARGS end

	table.insert( self.options, num, { text=text, id=command, access=access, callback=self.rconCallback } )

	return true
end


--[[
	Function: rconCallback
	
	The callback for options added with addRconOption. *DO NOT CALL DIRECTLY*

	Notes:

		This function does not take self as a parameter so it can be easily called back
]]
function ULib.Menu.rconCallback( userid, command )
	if command == ULib.ID_EXIT then return end

	_ServerCommand( command .. "\n" )
end


--[[
	Function: addPlayerOptions
	
	Adds player names to the menu. If a player is chosen, their userid is returned.

	Parameters:

		all_choice - *(Optional, defaults to false)* If true, an "All Players" will be shown. This has an id of ID_ALLPLAYERS.
		callback - *(Optional)* If a callback is specified, this callback will be used for this option instead of the one specified in new.

	Returns:

		True on success, nil and error message otherwise.
		
	Revisions:
	
		v1.2 - Fixed the allplayers id.
]]
function ULib.Menu:addPlayerOptions( all_choice, callback )
	if all_choice then
		table.insert( self.options, { text="All Players", id=ULib.ID_ALLPLAYERS, callback=callback } )
	end

	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) then
			table.insert( self.options, { text=_PlayerInfo( i, "name" ), id=i, callback=callback } )
		end
	end

	return true
end


--[[
	Function: showMenu
	
	Shows the menu to a user.

	Parameters:

		userid - The user to show the menu to, use 0 or nil for all players.
		page - *(Optional, defaults to 1)* The page to show to the user.
		timeout - *(Optional, defaults to 999)* How long it takes for the options to time out.

	Returns:

		True on success, nil and error message otherwise.
		
	Revisions:
	
		v1.01 - Now handles access correctly
]]
function ULib.Menu:showMenu( userid, page, timeout )
	page = page or 1
	timeout = timeout or 999
	if not ULib.checkParams( { userid, page }, { "number", "number" } ) then return nil, ULib.ERR_ARGS end
	if userid ~= 0 and not _PlayerInfo( userid, "connected" ) then return nil, "Player isn't connected" end
	if not self.skin.max_options or self.skin.max_options < 4 then 
		return nil, "You must have at least 4 in max options (Or you did not specify max options)"
	end

	-- Error checking done, let's get on with it!
	local max_options = self.skin.max_options -- Make things easier for us
	local page_options = {} -- This will hold our options
	local pageopts = max_options - 3 -- The number of options per page, excluding the first and last page. (-3 for next, last, and exit)
	local offset = (page - 1) * pageopts -- How many options to offset ourselves for this page.
	if page > 1 then offset = offset + 1 end -- To account for the first page (doesn't have last page option)

	for i=1, pageopts do -- Let's fill in our options
		if not self.options[ offset + i ] then
			break -- We're out of options! (Literally)
		end

		page_options[ i ] = self.options[ offset + i ]
		if userid ~= 0 and page_options[ i ].access then -- Don't bother with access sent to all players. (What would be the point in that?)
			page_options[ i ].disabled = not ulx.ucl:query( userid, page_options[ i ].access ) -- Flag disabled as need be.
		end
	end

	if page > 1 then -- Fill the next space with "last page" option.
		page_options[ pageopts + 1 ] = { text=self.skin.prev, id=ULib.ID_PREV }
	elseif self.options[ pageopts + 1 ] then -- It's the first page, if there are enough options, just fill this space up with another option.
		page_options[ pageopts + 1 ] = self.options[ pageopts + 1 ]
	end

	local next_page = offset + pageopts + 1 -- The offset for the next page.
	if page == 1 then next_page = next_page + 1 end -- To account for the first page having an extra option. (Do I really need to keep saying this?)

	if self.options[ next_page ] then -- If there's enough options for a next page, make a next page option!
		if not self.options[ next_page + 1 ] then -- If there's only one option that would be shown on the next page, put it on here instead.
			page_options[ pageopts + 2 ] = page_options[ pageopts + 1 ] -- Switch places with "last page" option.
			page_options[ pageopts + 1 ] = self.options[ next_page ]
		else -- Otherwise, put the next page option in.
			page_options[ pageopts + 2 ] = { text=self.skin.next, id=ULib.ID_NEXT }
		end
	end

	page_options[ max_options ] = { text=self.skin.exit, id=ULib.ID_EXIT } -- The final option is exit.

	local func_name = self.title .. userid
	_G[ func_name ] = function ( userid, option, time ) -- TODO: Use a timer to call this function on timeout.
		if option == 0 then option = 10 end -- Make it compatible
		if not page_options[ option ] or page_options[ option ].disabled then -- Ignore invalid input
			if self.skin.bind_options then
				timeout = timeout - time -- Note this is using a variable outside this callback's scope so it can track multiple decrements.
				_PlayerOption( userid, func_name, timeout )
			end
			return
		end

		self.skin.hideMenu( userid, page_options, page, self )

		if page_options[ option ].id == ULib.ID_PREV and page > 1 then
			self:showMenu( userid, page - 1 )
		elseif page_options[ option ].id == ULib.ID_NEXT then
			self:showMenu( userid, page + 1 )
		else -- It's an option we need to call a callback for. (NOTE THAT EXIT IS PASSED)
			if page_options[ option ].callback then -- They have a different callback
				page_options[ option ].callback( userid, page_options[ option ].id, time, page_options[ option ].text )
			else
				self.callback( userid, page_options[ option ].id, time, page_options[ option ].text )
			end
		end

		return
	end

	ULib.addTimer( timeout, 1, function() _G[ func_name ] = nil end ) -- Remove callback function

	if userid ~= 0 then
		if self.skin.bind_options then
			_PlayerOption( userid, func_name, timeout )
		end
		self.skin.showMenu( userid, self.title, page_options, timeout, page, self )
	else
		for i=1, _MaxPlayers() do
			if _PlayerInfo( i, "connected" ) then
				_PlayerOption( i, func_name, timeout )
				self.skin.showMenu( i, self.title, page_options, timeout, page, self )
			end
		end
	end

	return true
end








--[[
	Class: VoteMenu

	This handles ULib's voting menu.
]]

ULib.VoteMenu = {}
setmetatable( ULib.VoteMenu, ULib.VoteMenu )
ULib.VoteMenu.__index = ULib.Menu

--[[
	Function: new
	
	Creates a new vote menu.

	Parameters:

		title - The title of the menu.
		callback - The function to callback.
		skin - *(Optional, defaults to ULib.default_skin)* The skin to use for the menu.
		t - *(Optional)* A table to use to start out with.

	Returns:

		A new instantiation of the Menu class.
]]
function ULib.VoteMenu:new( title, callback, skin, t )
	t = t or {}
	local function reroute( userid, optionid, time, text ) t:userVote( userid, optionid, time, text ) end
	t = ULib.Menu:new( title, reroute, skin, t )
	setmetatable( t, t )
	t.__index = ULib.VoteMenu
	t.finalcallback = callback
	t.results = {}

	return t
end


--[[
	Function: userVote
	
	Handles a user's vote. *DO NOT CALL DIRECTLY*

	Parameters:

		userid - The userid making the vote
		optionid - The optionid chosen
		time - The time taken to choose
		
	Revisions:
	
		v1.2 - Fixed: wasn't counting votes right
]]
function ULib.VoteMenu:userVote( userid, optionid, time, text )
	if not self.results[ optionid ] then
		self.results[ optionid ] = { num=1, voters={ userid }, text=text }
	else
		if ULib.isInTable( self.results[ optionid ].voters, userid ) then return end -- Ignore vote stuffers... though I'm not sure how they'd get the chance.
		self.results[ optionid ].num = self.results[ optionid ].num + 1
		table.insert( self.results[ optionid ].voters, userid )
	end

	self.total_votes = self.total_votes + 1
	self.skin.updateVote( userid, optionid, time, text, self )

	if self.total_votes >= self.total_voters then self:endVote() end -- Allow votes to end early.
end


--[[
	Function: showVote
	
	Shows a vote

	Parameters:

		userid - The user to show the vote to. Use 0 or nil for all users
		time - *(Optional, defaults to 20)* The time in seconds to show the vote.

	Note:
	
		This will reset the vote results, so show the menu to everyone that needs to see it at the same time.
]]
function ULib.VoteMenu:showVote( userid, time )
	time = time or 20

	self.results = {}
	self:showMenu( userid, 1, time )
	self.total_voters = self.total_voters or 0
	
	if userid ~= 0 then
		self.total_voters = self.total_voters + 1
		self.skin.startVote( userid, time, self )
	else
		for i=1, _MaxPlayers() do
			if _PlayerInfo( i, "connected" ) then
				self.total_voters = self.total_voters + 1
				self.skin.startVote( i, time, self )
			end
		end
	end

	self.timerid = ULib.addTimer( time, 1, self.endVote, self )
	self.total_votes = 0
end


--[[
	Function: endVote
	
	Ends a vote
]]
function ULib.VoteMenu:endVote()
	if not self.timerid then return end -- Already handled
	
	local max = 0
	local winner = nil
	for k, v in pairs( self.results ) do
		if k ~= ULib.ID_EXIT then -- Ignore exits
			if v.num > max then
				winner = k
				max = v.num
			end
		end
	end
	local winning_id
	local winning_text
	if winner then
		winning_id = winner
		winning_text = self.results[ winner ].text
	end

	self.skin.endVote( winning_id, winning_text, self.total_votes, self.total_voters, self )
	self.finalcallback( winning_id, winning_text, self.total_votes, self.total_voters )

	self.timerid = nil
	self.total_votes = 0
	self.total_voters = 0
end