--[[
	Title: Physics Helpers
]]


--[[
	Function: applyAccel

	Parameters:
	
		entid - The entity to apply the acceleration to
		magnitude - The amount of acceleration ( Use nil if the magnitude is specified in the direction )
		direction - The direction to apply the acceleration in ( if the magnitude is part of the direction, specify nil for the magnitude )
		dTime - *(Optional, defaults to 1)* The time passed since the last update in seconds ( IE: 0.5 for dTime would only apply half the acceleration )
]]
function ULib.applyAccel( entid, magnitude, direction, dTime )
	if dTime == nil then dTime = 1 end
	
	if magnitude ~= nil then
		vecNormalize( direction )
	else
		magnitude = 1
	end
	
	-- Times it by the time elapsed since the last update.
	local accel = magnitude * dTime
	-- Convert our scalar accel to a vector accel
	accel = vecMul( direction, accel )
			
	if _EntGetMoveType( entid ) == MOVETYPE_VPHYSICS then
		-- a = f/m , so times by mass to get the force.
		local force = vecMul( accel, _phys.GetMass( entid ) )
		_phys.ApplyForceCenter( entid, force )
	else
		-- newvelocity = oldvelocity + accel
		local newVel = vecAdd( _EntGetVelocity( entid ), accel )
		_EntSetVelocity( entid, newVel )
	end
end


--[[
	Function: applyForce

	Parameters:
	
		entid - The entity to apply the force to
		magnitude - The amount of force ( Use nil if the magnitude is specified in the direction )
		direction - The direction to apply the force in ( if the magnitude is part of the direction, specify nil for the magnitude )
		dTime - *(Optional, defaults to 1)* The time passed since the last update in seconds ( IE: 0.5 for dTime would only apply half the force )
]]
function ULib.applyForce( entid, magnitude, direction, dTime )
	if dTime == nil then dTime = 1 end
	
	if magnitude ~= nil then
		vecNormalize( direction )
	else
		magnitude = 1
	end

	-- Times it by the time elapsed since the last update.
	local force = magnitude * dTime
	-- Convert our scalar force to a vector force
	force = vecMul( direction, force )
			
	if _EntGetMoveType( entid ) == MOVETYPE_VPHYSICS then
		_phys.ApplyForceCenter( entid, force )
	else
		-- Because we're not dealing with objects that have vphysics, they might not have a mass. This would cause errors, let's catch them here.
		local mass = _phys.GetMass( entid )
		if not mass then
			mass = 1
			ULib.debug( "applyForce was called with a non-physics entity that doesn't have a mass. To continue calculations, we're assuming it has a mass of one. This could very well produce unrealistic looking physics!\n")
		end
		-- f = m*a, so divide it by mass to get the accel
		local accel = vecMul( force, 1/mass )
		-- newvelocity = oldvelocity + accel
		local newVel = vecAdd( _EntGetVelocity( entid ), accel )
		_EntSetVelocity( entid, newVel )
	end
end


--[[
	Function: applyAccelInCurDirection
	
	Applies an acceleration in the entities current *velocity* direction ( not the entity's heading ). See <applyAccel>.
        Basically makes the entity go faster or slower ( if a negative magnitude is passed ).

	Parameters:

		entid - The entity to apply the force to
		magnitude - The amount of acceleration
		dTime - *(Optional, defaults to 1)* The time passed since the last update in seconds ( IE: 0.5 for dTime would only apply half the acceleration )
]]
function ULib.applyAccelInCurDirection( entid, magnitude, dTime )
	local direction = vecNormalize( _EntGetVelocity( entid ) )
	ULib.applyAccel( entid, magnitude, direction, dTime )
end


--[[
	Function: applyForceInCurDirection
	
	Applies a force in the entities current *velocity* direction ( not the entity's heading ). See <applyForce>. 
        Basically makes the entity go faster or slower ( if a negative magnitude is passed ).

	Parameters:

		entid - The entity to apply the force to
		magnitude - The amount of force
		dTime - *(Optional, defaults to 1)* The time passed since the last update in seconds ( IE: 0.5 for dTime would only apply half the force )
]]
function ULib.applyForceInCurDirection( entid, magnitude, dTime )
	local direction = vecNormalize( _EntGetVelocity( entid ) )
	ULib.applyForce( entid, magnitude, direction, dTime )
end