--[[
	Class: Database

	The Database "class" is exactly what you would think.... a database.
	The only major difference from just a regular table is that the Database "class" has saving and loading capabilities.
	More functionality may be added at a later date.

        Example:

		: o = Database:new()
		: o.test = true
		: o:save( "test.lua" )
		: o = nil
		: _Msg( tostring( o ) ) -- Output: nil
		: o = Database:newFromFile( "test.lua" )
		: _Msg( tostring( o.test ) ) -- Output: true
]]
ULib.Database = {}

--[[
	Function: new
	
	Creates a new Database.

	Parameters:

		t - *(Optional)* A table to use to start out with.

	Returns:

		A new instantiation of the Database class.
]]
function ULib.Database:new( t )
	t = t or {}   -- Create object if user does not provide one
	setmetatable( t, self ) -- If you're unfamiliar with metatables read <http://www.lua.org/pil/13.html>
	self.__index = self

	return t
end

--[[
	Function: newFromFile
	
	Creates a new Database from a file. If the file does not exist, it generates an error.

	Parameters:

		file - The file to load from, relative to the gmod9 folder.

	Returns:

		A new instantiation of the Database class, filled with the information from the file.
]]
function ULib.Database:newFromFile( file )
	if not ULib.checkParam( file, "string" ) or not _file.Exists( file ) then return nil, ULib.ERR_ARGS end

	_OpenScript( file )
	local b = o
	o = nil -- This allows for garbage collection if they delete the b reference later
	self:new( b )

	return b
end

--[[
	Function: save

	Saves the Database to a file.

	Parameters:

		file - The file to save to.
]]
function ULib.Database:save( file )
	_file.Write( file, "o = " .. ULib.serialize( self, ULib.SPACE_METHOD, "o" ) )
end

















--[[
	Class: SetDatabase

	The Database class holds "sets" of information. When a set that does not yet exist is called for from
	Database, Database creates a new set from default parameters for immediate use.

	For example, if you want a player database indexed by steamid storing information on the players such
	as number of props spawned, which would default to 0, and numOfWarningsTillBan, which might default
	to 3, you could use Database. Then, every time you reference a new steamid, a new set with the default
	information will be created. 
        
	Example:

		: players = Database:new()
		: players:setDefault( "numProps", 0 )
		: players:setDefault( "numOfWarningsTillBan", 3 )
		: -- Player with steamid STEAM_0:1:1234567 spawned a prop, so we need to increment his numProps
		: players[ "STEAM_0:1:1234567" ].numProps = players[ "STEAM_0:1:1234567" ].numProps + 1
		Since a new key has been referenced, this statement will create a new set for the user
		: _Msg( players[ "STEAM_0:1:1234567" ].numProps ) -- Output: 1
		: _Msg( players[ "STEAM_0:1:1234567" ].numOfWarningsTillBan ) -- Output: 3
]]
ULib.SetDatabase = {}
setmetatable( ULib.SetDatabase, ULib.SetDatabase ) -- If you're unfamiliar with metatables read <http://www.lua.org/pil/13.html>
ULib.SetDatabase.__index = ULib.Database

--[[
	Function: new
	
	Creates a new SetDatabase.

	Parameters:

		t - *(Optional)* A table with defaults for the new sets.

	Returns:

		A new instantiation of the SetDatabase class.
]]
function ULib.SetDatabase:new( t )
	local defaults = t or {}
	t = { __defaults=defaults }
	t = ULib.Database:new( t )   -- Create the object
	setmetatable( t, t ) -- If you're unfamiliar with metatables read <http://www.lua.org/pil/13.html>
	--[[
		Function: __index
		
		Part of SetDatabase's metatable, this is the function that makes new sets when they're needed. It's called automatically when a non-existant key is referenced.

		Parameters:

			table - The table from which the non-existant key was referenced
			key - The non-existant key that we need to fill in with the defaults
			
		Returns:
			
			*table[key]* filled with the defaults

		See Also:

			http://www.lua.org/pil/13.html -- Provides an explanation of metatables

			http://www.lua.org/pil/13.4.1.html -- Provides an explanation of the __index metamethod
	]]
		
	function t.__index( table, key )
		if self[ key ] then
			return self[ key ] -- To allow for inheritance
		end
		table[ key ] = copyTable( t.__defaults )
		return table[ key ]
	end	
	
	return t
end

--[[
	Function: setDefault

	Adds or changes a default key and value for new sets. 
	Note that the function will run through all current sets and add the value you specify to them.

	Parameters:

		k - The key for the default value
		v - The default value
		override - *(Optional, defaults to false)* If true, when setDefault() runs through the sets adding the
		value you specified and it finds that the key already exists, it will overwrite the existing value.
		Otherwise, it leaves the existing value.
]]
function ULib.SetDatabase:setDefault( k, v, override )
	self.__defaults[ k ] = v

	for key in pairs( self ) do
		if key ~= "__index" and key ~= "__defaults" then -- Skip over the utility functions
			if self[ key ][ k ] == nil or override then -- If it's nil fill it with the default ( don't overwrite unless override is on )
				self[ key ][ k ] = v
			end
		end
	end
end

--[[
	Function: setDefaultTable

	Calls <setDefault()> for each key/value in the given table

	Parameters:

		table - The table with the wanted keys and values
		override - *(Optional, defaults to false)* Passed to <setDefault()>
		
	Example:
	
		:setDefaultTable{ "numProps"=0, "numOfWarningsTillBan"=3 }
		Note the function call above uses the table function caller, it is the same as:
		:setDefaultTable( { "numProps"=0, "numOfWarningsTillBan"=3 } )

	See Also:

		<setDefault()>
]]
function ULib.SetDatabase:setDefaultTable( table, override )
	for k, v in pairs( table ) do
		self:setDefault( k, v, override )
	end
end

--[[
	Function: save

	Saves the SetDatabase to a file, including the defaults.

	Parameters:

		file - The file to save to.
		prefix - *(Optional)* If you only want variables with a certain prefix within the sets to be saved, use this.
]]
function ULib.SetDatabase:save( file, prefix )

        local t = ULib.copyTable( self )
	if prefix then -- If we only want to save certain prefixes

		for key in pairs( t ) do
			if key ~= "__index" then -- Skip over the utility functions
				if type( t[ key ] ) == "table" then -- Skip over nontables ( They should all be tables )
					for key2 in pairs( t[ key ] ) do
						if string.sub( tostring( key2 ), 1, string.len( prefix ) ) ~= prefix then
							t[ key ][ key2 ] = nil
						end
					end
				end
			end
		end
		
	end
	
	t = ULib.Database:new( t )
	return t:save( file )
end


---------------------------------------
--88888888888888888888888888888888888--
---------------------------------------


--[[
	Table: db

	This database saves every 30 seconds. This database can be used for permanent storage.
]]
	if _file.Exists( ULib.DB_FILE ) then ULib.db = ULib.Database:newFromFile( ULib.DB_FILE )
	else ULib.db = ULib.Database:new() end


--[[
	Function saveDb

	This function saves the database every 30 seconds.
]]
	function ULib.saveDb()
		ULib.db:save( ULib.DB_FILE )
	end
	ULib.addTimer( 30, 0, ULib.saveDb )