--[[
	Title: Download

	Controls what files are downloaded for what maps
]]


--[[
	Function: getDownloads

	Retrieves a table containing all files to be downloaded for a map

	Parameters:

		map - The name of the map to look for, without an extension

	Returns:
	
		A table containing the files to be downloaded by clients for the map, nil and error otherwise.
]]
function ULib.getDownloads( map )
	if not ULib.checkParam( map, "string" ) then return nil, ULib.ERR_ARGS end

	local prefix = "maps/" .. map
	if not _file.Exists( prefix .. ".bsp" ) then return nil, "No such map" end

	local t = {}
	if not _file.Exists( prefix .. ".res" ) then return t end -- No downloads
	t = ULib.tableToLower( ULib.parseKeyValues( _file.Read( prefix .. ".res" ) ) )
	if not t.resources then return nil, "Invalid res file" end

	local o = {} -- This will hold the final table
	for k in pairs( t.resources ) do
		table.insert( o, k ) -- Reverse keys and values
	end
	
	return o
end


--[[
	Table: gDownloads
	
	This table supports the temporary forced download cleanup.
]]
local gDownloads = {}


--[[
	Function: forceDownload

	Forces players to download a file on connect.

	Parameters:

		file - The file to force download, relative to the gmod9 folder.
		perm - *(Optional, defaults to false)* If true, the file will not be removed on mapchange. *NOT RECOMMENDED TO SET TO TRUE*
		no_restart - *(Optional, defaults to false)* If false, the server will restart so it can load the forced download.

	Returns:
	
		True on success, nil and error otherwise.
]]
function ULib.forceDownload( file, perm, no_restart )
	if not ULib.checkParam( file, "string" ) then return nil, ULib.ERR_ARGS end

	local res = "maps/" .. _GetCurrentMap() .. ".res"
	gDownloads[ string.lower( file ) ] = true

	local t = { resources={} }
	if _file.Exists( res ) then
		t = ULib.tableToLower( ULib.parseKeyValues( _file.Read( res ) ) )
		if not t.resources then return nil, "Invalid res file" end
	end

	if t.resources[ file ] then return end -- No need to continue, it's already there.

	local value = "ULib" -- We know it's temp by this value.
	if perm then value = "file" end

	t.resources[ file ] = value

	local str, err = ULib.makeKeyValues( t )
	if not str then return nil, err end -- Pass on the error
	_file.Write( res, str )

	if not no_restart then
		ULib.addTimer( 0.1, 1, _ServerCommand, "changelevel " .. _GetCurrentMap() .. "\n" ) -- This will give time to add other files.
	end

	return true
end


--[[
	Function: forceDownloadFolder

	Forces players to download all files in a folder on connect.

	Parameters:

		folder - The folder to force download, relative to the gmod9 folder.
		perm - *(Optional, defaults to false)* If true, the file will not be removed on mapchange. *NOT RECOMMENDED TO SET TO TRUE*
		no_restart - *(Optional, defaults to false)* If false, the server will restart so it can load the forced download.
		recurse - *(Optional, defaults to false)* If true, it will also download folders within the folder

	Returns:
	
		True on success, nil and error otherwise.
]]
function ULib.forceDownloadFolder( folder, perm, no_restart, recurse )
	local files, err = ULib.filesInDir( folder, recurse )

	if not files then return nil, err end

	for _, v in ipairs( files ) do
		local ret, err ULib.forceDownloadFile( v, perm, no_restart )
		if not ret then return nil, err end
	end
end


-- This code will be run every map change, it cleans out .res files, deleting any old information.
local hookid
local function cleanRes( timed )
	if type( timed ) ~= "boolean" then -- This will make it so the function is run 120 seconds after the first player joins. We're in no hurry to execute this, it's pretty much optional.
		UnHookEvent( hookid )
		ULib.addTimer( 120, 1, cleanRes, true )
		return
	end

	if _file.Exists( "maps/" .. _GetCurrentMap() .. ".res" ) then -- Let's clean up temp download options.
		local res = "maps/" .. _GetCurrentMap() .. ".res"

		t = ULib.tableToLower( ULib.parseKeyValues( _file.Read( res ) ) )
		if t.resources then -- Make sure it's valid
			for k, v in pairs ( t.resources ) do
				if v == "ULib" and not gDownloads[ string.lower( k ) ] then -- If this hasn't been added again this map change to the temp list, remove it
					t.resources[ k ] = nil
				end
			end

			local str = ULib.makeKeyValues( t )
			if str then _file.Write( res, str ) end
		end
	end
end
hookid = HookEvent( "eventPlayerActive", cleanRes )