--[[
	Title: List

	This file contains a basic ULib.List class as well as Queue and Stack.
	
	Revisions:
	
		v1.1 - Initial
		v1.2 - Fixed. Don't try to use prior to this version.
]]


--[[
	Class: List

	The List "class" is exactly what you would think.... a ULib.List.
	It features pushing and popping left and right.
]]
ULib.List = {}

--[[
	Function: new
	
	Creates a new ULib.List.

	Returns:

		A new instantiation of the ULib.List class.
]]
function ULib.List:new()
	local o = {first = 1, last = 0}

	setmetatable(o, self)
	self.__index = self

	return o
end


--[[
	Function: pushLeft
	
	Pushes a value onto the end of the List

	Parameters:

		value - The value of any type to push.

	Returns:

		True on success, nil and error message otherwise.
]]
function ULib.List:pushLeft( value )
	local first = self.first - 1
	self.first = first
	self[ first ] = value
	
	return true
end


--[[
	Function: pushRight
	
	Pushes a value onto the beginning of the List

	Parameters:

		value - The value of any type to push.

	Returns:

		True on success, nil and error message otherwise.
]]
function ULib.List:pushRight( value )
	local last = self.last + 1
	self.last = last
	self[ last ] = value
	
	return true
end


--[[
	Function: popLeft
	
	Pops a value from the end of the List

	Returns:

		value on success, nil and error message otherwise.
]]
function ULib.List:popLeft()
	local first = self.first
	if first > self.last then return nil, "ULib.List is empty" end

	local value = self[ first ]
	self[ first ] = nil        -- to allow garbage collection
	self.first = first + 1

	return value
end


--[[
	Function: popRight
	
	Pops a value from the beginning of the List

	Returns:

		value on success, nil and error message otherwise.
]]
function ULib.List:popRight()
	local last = self.last
	if self.first > last then return nil, "ULib.List is empty" end

	local value = self[ last ]
	self[ last ] = nil         -- to allow garbage collection
	self.last = last - 1

	return value
end


--[[
	Function: isEmpty
	
	Tells you if the List is empty

	Returns:

		True if empty, false otherwise.
]]
function ULib.List:isEmpty()
	if self.first > self.last then
		return true
	else
		return false
	end
end



--[[
	Class: Queue

	The Queue "class" is exactly what you would think.... a queue.
	It features pushing (to the right) and popping (from the left).
	A queue stores data in FIFO (First In, First Out).
	You can use any of the functions from the <ULib.List> class, but
	obviously you don't want to use its push/pop functions (breaks
	the queue).
]]
ULib.Queue = ULib.List:new()


--[[
	Function: push
	
	Pushes a value onto the queue

	Parameters:

		value - The value of any type to push.

	Returns:

		True on success, nil and error message otherwise.
]]
function ULib.Queue:push( value )
	return self:pushRight( value )
end


--[[
	Function: pop
	
	Pops a value from the queue

	Returns:

		value on success, nil and error message otherwise.
]]
function ULib.Queue:pop()
	return self:popLeft()
end



--[[
	Class: Stack

	The Stack "class" is exactly what you would think.... a stack.
	It features pushing (to the left) and popping (from the left).
	A stack stores data in LIFO (Last In, First Out).
	You can use any of the functions from the <ULib.List> class, but
	obviously you don't want to use its push/pop functions (breaks
	the stack).
]]
ULib.Stack = ULib.List:new()


--[[
	Function: push

	Pushes a value onto the stack

	Parameters:

		value - The value of any type to push.

	Returns:

		True on success, nil and error message otherwise.
]]
function ULib.Stack:push( value )
	return self:pushLeft( value )
end


--[[
	Function: pop
	
	Pops a value from the stack

	Returns:

		value on success, nil and error message otherwise.
]]
function ULib.Stack:pop()
	return self:popLeft()
end