--[[
	Class: Log
	
	The log class is used to easily create logs with sequential numbering.
	
	Example:
		
		: log = ULib.Log:new("test", "logs/testlogs", 2)
		: log:write("This is a test. ")
		: log:writeln("This will be on the same line.")
		: log:writeln("This is on a new line.")
		: log:writeln("This is an important line." )
		: log:flush() --Flush the buffer so we make sure the last message gets written
		: log:writeln("This is a timestamped test.", true)
		: log:write("This is another timestamped test. ", true)
		: log:writeln("This is on the same line.")
		
	Example Output:
		
		This is a test. This will be on the same line.
		This is on a new line.
		This is an important line.
		00:01:04] This is a timestamped test.
		00:01:20] This is another timestamped test. This is on the same line.
		
	Revisions:
	
		v1.2 - Initial
]]
ULib.Log = {}

--[[
	Function: new
	
	Creates a new Log
	
	Parameters:
		
		prefix - Prefix of log file. (Ex. "test" => "test_1.log")
		dir - Directory to save logs to
		delay - *(Optional, defaults to 2)* Time in seconds to wait between each write to file. 0 to write instantly.

	Returns:
		
		A new instantiation of the List class, nil and error otherwise.
		
	Revisions:
	
		v1.2 - Initial
]]
function ULib.Log:new(prefix, dir, delay)
	delay = delay or 2
	if not ULib.checkParams( { prefix, dir, delay }, { "string", "string", "number" } ) then return nil, "Incorrect parameters specified!" end
	if not _GetCurrentMap() or _GetCurrentMap() == "" then return nil, "No map loaded" end
	if not string.sub(dir,-1) ~= "/" then dir = dir.."/" end
	if not _file.Exists(dir) then
		_file.CreateDir(dir)
	elseif not _file.IsDir(dir) then
		return nil, "\""..dir.."\" is not a directory!";
	end
	local t = {}
	t.prefix = prefix
	t.dir = dir
	t.buffer = {}
	t.delay = delay
	local high = 0
	for _,v in ipairs(_file.Find(dir.."*")) do
		if string.sub(v,1,string.len(prefix)) == prefix then
			local temp = tonumber(string.sub(v,string.len(prefix)+2,-5))
			if temp > high then high = temp end
		end
	end
	t.filename = prefix.."_"..tostring(high+1)..".log"
	setmetatable( t, self )
	self.__index = self
	
	if t.delay ~= 0 then AddTimer(t.delay, 0, t.flush, t) end
	
	return t
end
--[[
	Function: write
	
	Writes to the log.
	
	Parameters:
	
		str - The string to write to the log.
		ts - *(Optional, defaults to false)* If true, adds a timestamp to the start of the log string.

	Revisions:
	
		v1.2 - Initial
]]
function ULib.Log:write(str, ts)
	if not ULib.checkParam(str, "string") then return nil, "Incorrect parameters specified!" end
	if ts then str = ULib.getTimeStamp( _CurTime() ).."] "..str end
	callLogHooks(self.prefix, self.filename, str)
	table.insert(self.buffer, str)
	if self.delay == 0 then self:flush() end
end
--[[
	Function: writeln
	
	Writes a line to the log,
	
	Parameters:
		
		str - The string to write to the log.
		ts - *(Optional, defaults to false)* If true, adds a timestamp to the start of the log string.

	Revisions:
	
		v1.2 - Initial
]]
function ULib.Log:writeln(str, ts)
	if not ULib.checkParam(str, "string") then return nil, "Incorrect parameters specified!" end
	self:write(str..ULib.NEWLINE, ts)
end

--[[
	Function: flush
	
	Flushes the buffer to the file.
	
	Revisions:
	
		v1.2 - Initial
]]
function ULib.Log:flush()
	if table.getn( self.buffer ) > 0 then -- Only write if there is something to write
		if _file.Exists(self.dir..self.filename) then
			_file.Write(self.dir..self.filename,_file.Read(self.dir..self.filename)..ULib.implode(self.buffer,""))
		else
			_file.Write(self.dir..self.filename,ULib.implode(self.buffer,""))
		end
		self.buffer = {}
	end
end

--[[
	Table: logHooks
	
	This table holds log hook functions.
	
	Revisions:
	
		v1.2 - Initial
]]
logHooks = {}

--[[
	Function: hookLog
	
	This function adds a log hook. 
	
	Parameters:
	
		fn - The function to callback on log write.
		
	Revisions:
	
		v1.2 - Initial
]]
function ULib.hookLog(fn)
	if not ULib.checkParam( fn, "function" ) then return nil, ULib.ERR_ARGS end
	if not ULib.isInTable( logHooks, fn ) then return nil, "Function already hooked!" end
	table.insert(logHooks, fn)
end

--[[
	Function: unHookLog
	
	This function unhooks a log hook.
	
	Parameters:
	
		fn - The function to unhook
		
	Revisions:
	
		v1.2 - Initial
]]
function ULib.unHookLog(fn)
	if not ULib.checkParam( fn, "function" ) then return nil, ULib.ERR_ARGS end
	if not ULib.isInTable( logHooks, fn ) then return nil, "Function not hooked!" end
	for k, v in ipairs(logHooks) do
		if v == fn then
			table.remove(logHooks, k)
			return
		end
	end
end

--[[
	Function: callLogHooks
	
	Calls the log hooks. *DO NOT CALL DIRECTLY*
	
	Log hooks are called with parameters prefix, filename, and str
	
	Revisions:
	
		v1.2 - Initial
]]
function callLogHooks(prefix, filename, str)
	for _,v in ipairs(logHooks) do
		v(prefix, filename, str) 
	end
end
