--[[
	Title: Tables

	This file contains table helper functions.
]]


--[[
	Function: copyTable

	Copies a table. If you simply use *newTable = source* you'll quickly notice that the new table is pointing to the old table. copyTable() makes the new table independent of the old table.

	Parameters:

		source - The source table you wish to copy.

	Returns:

		The copied table, nil and error message otherwise.
]]
function ULib.copyTable( source )
	if not ULib.checkParam( source, "table" ) then return nil, ULib.ERR_ARGS end
	local target = {}
	for k, v in pairs( source ) do
		if type( v ) == "table" then
			target[ k ] = ULib.copyTable( source[ k ] )
		else
			target[ k ] = source[ k ]
		end
	end
	return target
end


--[[
	Function: isInTable

	Checks if a value is in a table. Checks *values, not keys*.

	Parameters:

		t - The table to check
		check - The value to check if it exists in t. Can be any type.
		recursive - *(Optional, default to false)* If true, it will check any subtables it comes across.

	Returns:

		True and the key of the value (value only if it's not recursive) if it's in the table, false if it's not. Nil and error message otherwise.
		
	Revisions:
	
		v1 - Improved recursive functionality
]]
function ULib.isInTable( t, check, recursive )
	local returned
	if not ULib.checkParam( t, "table" ) or not check then return nil, ULib.ERR_ARGS end

	for k, v in pairs( t ) do
		if v == check then return true, k end
		
		if type( v ) == "table" and recursive then returned = ULib.isInTable( v, check, recursive ) end
		if returned then return true end
	end
	
	return false
end


--[[
	Function: findInTable

	Finds a value in a table. As opposed to <isInTable()>, this function will *only* check numeric keys, and will return a number of where the value is.

	Parameters:

		t - The table to check
		check - The value to check if it exists in t. Can be any type.
		init - *(Optional, defaults to 1)* The value to start from.
		last - *(Optional, defaults to the length of the table)* The value to end at.
		recursive - *(Optional, default to false)* If true, it will check any subtables it comes across.

	Returns:

		The number of the key where check resides, false if none is found. If init > last it returns false as well. Nil and error message otherwise.
]]
function ULib.findInTable( t, check, init, last, recursive )
	if not ULib.checkParam( t, "table" ) or not check then return nil, ULib.ERR_ARGS end

	init = init or 1
	last = last or table.getn( t )

	if init > last then return false end

	for i=init, last do
		if t[ i ] == check then return i end

		if type( t[ i ] ) == "table" and recursive then return ULib.findInTable( v, check, 1, recursive ) end
	end

	return false
end


--[[
	Function: sharedValues

	Returns any values that are shared by two tables.

	Parameters:

		t1 - The first table.
		t2 - The second table.

	Returns:

		A table containing the shared values, false if there are no shared values. Nil and error message otherwise.
]]
function ULib.sharedValues( t1, t2 )
	if not ULib.checkParams( { t1, t2 }, { "table", "table" } ) then return nil, ULib.ERR_ARGS end
	local share = {}
	
	if table.getn( t1 ) < table.getn( t2 ) then -- Let's make sure we're looping thru the one with the least values
		t1, t2 = t2, t1
	end
	
	for _, v in pairs( t2 ) do
		if ULib.isInTable( t1, v ) then
			table.insert( share, v )
		end
	end
	
	if table.getn( share ) == 0 then return false end

	return share
end


--[[
	Function: mergeTable

	Merges the elements of one or more arrays together so that the values of one are appended to the end of the previous one.
	If the input arrays have the same string keys, then the later value for that key will overwrite the previous one. If, however, the arrays contain numeric keys, the later value will not overwrite the original value, but will be appended.
	If only one array is given, the numerical keys get reindexed from 1.

	Parameters:

		t - The first table
		... - Any number of extra tables to merge

	Returns:

		The resulting merged table.
]]
function ULib.mergeTable( t, ... )
	local t1 = {}
	local n = 1
	for k, v in pairs( t ) do
		if tonumber( k ) then
			t1[ n ] = t[ k ]
			n = n + 1
		else
			t1[ k ] = t[ k ]
		end
	end

	if arg then
        for k, v in pairs( arg[ 1 ] ) do
			if tonumber( k ) then
				t1[ n ] = arg[ 1 ][ k ]
				n = n + 1
			else
				t1[ k ] = arg[ 1 ][ k ]
			end
		end
	end

	if table.getn( arg ) > 1 then
		table.remove( arg, 1 )
		t1 = ULib.mergeTable( t1, unpack( arg ) )
	end

	return t1
end


--[[
	Function: removeEmptyValues

	This function will remove empty table values ( just spaces or "" ) from a table.

	Parameters:
		t - The table
	
	Returns:
		A copy of the table with the values remove.
]]
function ULib.removeEmptyValues( t )
	local o = ULib.copyTable( t )
	local offset = 0

	for k, v in pairs( t ) do
		if not string.find( v, "%S" ) then
			if type( k ) == "number" then
				table.remove( o, k - offset )
				offset = offset + 1
			else
				o[ k ] = nil
			end
		end
	end
	
	return o
end


--[[
	Function: tableToLower

	This function will make any string keys in the table lower case.

	Parameters:

		t - The table to use.
		recursive - *(Optional, default to false)* If true, it will change any subtables it comes across.

	Returns:

		A copy of the table with the keys lowercased.
		
	Note:
		
		This will have problems if the table has keys with multiple cases. IE, a key of "Amazing" and "aMazing", only one will survive.
]]
function ULib.tableToLower( t, recursive )
	if not ULib.checkParam( t, "table" ) then return nil, "Incorrect parameter" end
	local o = ULib.copyTable( t )
	local changes = {}
	for key, value in pairs( o ) do
		if type( key ) == "string" and string.lower( key ) ~= key then
			table.insert( changes, key )
		end

		if recursive and type( value ) == "table" then
			o[ key ] = ULib.tableToLower( o[ key ] )
		end

	end

	for _, key in ipairs( changes ) do -- We have to separate this because pairs() is confused otherwise.
		o[ string.lower( key ) ] = o[ key ]
		o[ key ] = nil
	end

	return o
end


--[[
	Function: tableToUpper

	This function will make any string keys in the table upper case.

	Parameters:

		t - The table to use.
		recursive - *(Optional, default to false)* If true, it will change any subtables it comes across.

	Returns:

		A copy of the table with the keys uppercased.

	Note:
		
		This will have problems if the table has keys with multiple cases. IE, a key of "Amazing" and "aMazing", only one will survive.
]]
function ULib.tableToUpper( t, recursive )
	if not ULib.checkParam( t, "table" ) then return nil, "Incorrect parameter" end
	local o = ULib.copyTable( t )
	local changes = {}
	for key, value in pairs( o ) do
		if type( key ) == "string" and string.upper( key ) ~= key then
			table.insert( changes, key )
		end

		if recursive and type( value ) == "table" then
			o[ key ] = ULib.tableToUpper( o[ key ] )
		end

	end

	for _, key in ipairs( changes ) do -- We have to separate this because pairs() is confused otherwise.
		o[ string.upper( key ) ] = o[ key ]
		o[ key ] = nil
	end
	
	return o
end

--[[
	Function: numValues

	Similar to table.getn, but this returns the TRUE number of elements in the table, not just the number of ordered values.

	Parameters:

		t - The table.

	Returns:

		The number of elements, nil and error message otherwise.
		
	Revisions:
	
		v1.1 - Initial
]]
function ULib.numValues( t )
	if not ULib.checkParam( t, "table" ) then return nil, ULib.ERR_ARGS end

	local num = 0
	for _, _ in pairs( t ) do
		num = num + 1
	end
	
	return num
end