--[[
	Title: UCL

	ULib's Access Control List
]]

ACCESS_ALL		=	nil
ACCESS_NONE		=	0
ACCESS_IMMUNITY 	= 	"a"
ACCESS_RESERVATION	=	"b"
ACCESS_KICK		=	"c"
ACCESS_BAN		=	"d"
ACCESS_SLAY		=	"e"
ACCESS_MAP		=	"f"
ACCESS_CVAR		=	"g"
ACCESS_ENT		=	"h"
ACCESS_CHAT		=	"i"
ACCESS_VOTE		=	"j"
ACCESS_PROP		=	"k"
ACCESS_RCON		=	"l"
ACCESS_PLAYER		=	"m"
ACCESS_MENU		=	"n"
ACCESS_RESPECTED	=	"R" -- Put in by popular demand in v1.2! :)

ACCOUNT_PLAYERNAME	=	"a"
ACCOUNT_CLANTAG		=	"b"
ACCOUNT_STEAMID		=	"c"
ACCOUNT_IP		=	"d"
ACCOUNT_STEAMLOGIN	=	"e"
ACCOUNT_PASSREQ		=	"f"

ULib.reverseAccess				=	{} -- This is used by concommand's dynamic access. Not we can't handle ACCESS_ALL
ULib.reverseAccess[ ACCESS_NONE ]		=	"ACCESS_NONE"
ULib.reverseAccess[ ACCESS_IMMUNITY ]		=	"ACCESS_IMMUNITY"
ULib.reverseAccess[ ACCESS_RESERVATION ]	=	"ACCESS_RESERVATION"
ULib.reverseAccess[ ACCESS_KICK ]		=	"ACCESS_KICK"
ULib.reverseAccess[ ACCESS_BAN ]		=	"ACCESS_BAN"
ULib.reverseAccess[ ACCESS_SLAY ]		=	"ACCESS_SLAY"
ULib.reverseAccess[ ACCESS_MAP ]		=	"ACCESS_MAP"
ULib.reverseAccess[ ACCESS_CVAR ]		=	"ACCESS_CVAR"
ULib.reverseAccess[ ACCESS_ENT ]		=	"ACCESS_ENT"
ULib.reverseAccess[ ACCESS_CHAT ]		=	"ACCESS_CHAT"
ULib.reverseAccess[ ACCESS_VOTE ]		=	"ACCESS_VOTE"
ULib.reverseAccess[ ACCESS_PROP ]		=	"ACCESS_PROP"
ULib.reverseAccess[ ACCESS_RCON ]		=	"ACCESS_RCON"
ULib.reverseAccess[ ACCESS_PLAYER ]		=	"ACCESS_PLAYER"
ULib.reverseAccess[ ACCESS_MENU ]		=	"ACCESS_MENU"
ULib.reverseAccess[ ACCESS_RESPECTED ]		=	"ACCESS_RESPECTED"

ULib.access = {} -- Used by concommand's dynamic access. I realize that it could use the defines above, but this is much safer.
ULib.access.ACCESS_NONE		=	ACCESS_NONE
ULib.access.ACCESS_IMMUNITY 	= 	ACCESS_IMMUNITY
ULib.access.ACCESS_RESERVATION	=	ACCESS_RESERVATION
ULib.access.ACCESS_KICK		=	ACCESS_KICK
ULib.access.ACCESS_BAN		=	ACCESS_BAN
ULib.access.ACCESS_SLAY		=	ACCESS_SLAY
ULib.access.ACCESS_MAP		=	ACCESS_MAP
ULib.access.ACCESS_CVAR		=	ACCESS_CVAR
ULib.access.ACCESS_ENT		=	ACCESS_ENT
ULib.access.ACCESS_CHAT		=	ACCESS_CHAT
ULib.access.ACCESS_VOTE		=	ACCESS_VOTE
ULib.access.ACCESS_PROP		=	ACCESS_PROP
ULib.access.ACCESS_RCON		=	ACCESS_RCON
ULib.access.ACCESS_PLAYER	=	ACCESS_PLAYER
ULib.access.ACCESS_MENU		=	ACCESS_MENU
ULib.access.ACCESS_RESPECTED	=	ACCESS_RESPECTED


--[[
	Variable: PASSWORD_TIME

	Default password time for authenticating
]]
local PASSWORD_TIME = 30
ULib.CONVAR( "ulib_passtime", PASSWORD_TIME, "This sets how long people have to enter their pass, if required.", ACCESS_KICK )


--[[
	Variable: PASSWORD_TIMEOUT
	
	Default password timeout in minutes (at which point they'll have to re-enter their password).
]]
local PASSWORD_TIMEOUT = 60
ULib.CONVAR( "ulib_passtimeout", PASSWORD_TIME, "This sets how long before people have to re-enter their password", ACCESS_KICK )


--[[
	Section: Format

	This section will tell you how to format your strings/files for UCLs.

	Format of admin account--
	"<name|ip|steamid|clantag|steamlogin>" "<password>" "<access flags>" "<account flags>"
	
	Access flags--
	a - immunity (can't be kicked/baned/slayed/slaped and affected by other commmands)
	b - reservation (can join on reserved slots)
	c - kick access
	d - ban and unban access
	e - slay and slap access
	f - map access
	g - cvar access
	h - ent access (could crash the server!)
	i - misc chat access
	j - vote access
	k - prop access
	l - rcon access
	m - player control access (things that affect clients)
	n - menu access
	R - Respected
	Other letters may be used for additional plugins (access flags are caps-sensitive).
	
	Account flags--
	a - this is a playername
	b - this is a clan tag (partial playername)
	c - this is a steamid
	d - this is an ip
	e - this is a steam login (the username they log into steam with)
	f - disconnect player on invalid password. This MUST be used with another account flag. (It's the only flag that can be used with others)
	    (gives them 30 seconds to enter their pass, by default)
	
	Password--
	When asked for the password, use "_pw <password>" in console
	
	Examples of admin accounts--
	"STEAM_0:0:123456" "" "abcdefghijklmnopqrstuv" "c"
	"123.45.67.89" "" "abcdefghijklmnopqrstuv" "d"
	"My Name" "my_password" "abcdefghijklmnopqrstuv" "a"
]]


--[[
	Table: ucl

	Holds all of the ucl variables and functions
]]
ULib.ucl = {}


--[[
	Function: new()

	Creates a new, empty UCL object.

	Returns:

		The new ucl object.
]]
function ULib.ucl:new()
	local t = {}
	t.playernames = {}
	t.clantags = {}
	t.steamids = {}
	t.ips = {}
	t.steamlogins = {}
	t.authed = {}
	t.awaitingauth = {}
	t.callbacks = {}

	setmetatable( t, self )
	self.__index = self

	local routeUserActive = function ( name, userid )
		t.authed[ userid ] = nil
		t.awaitingauth[ userid ] = nil
		return t:probe( userid )
	end
	HookEvent( "eventPlayerActive", routeUserActive )

	local userDisconnect = function ( name, userid )
		t.authed[ userid ] = nil
		t.awaitingauth[ userid ] = nil
	end
	HookEvent( "eventPlayerDisconnect", userDisconnect )

	local routePw = function ( userid, args )
		return t:checkPass( userid, args )
	end
	CONCOMMAND( "_pw", routePw )

	ULib.db.ucl = ULib.db.ucl or {}

	-- Clean up our database
	local db = ULib.db.ucl

	local passtimeout = tonumber( ULib.getCvar( "ulib_passtimeout" ) )
	passtimeout = passtimeout or PASSWORD_TIMEOUT -- If there's something invalid in the value, fallback on default.
	passtimeout = passtimeout * 60 -- Convert to minutes

	for k, v in pairs( db ) do
		if ULib.db.uptime > v.time + passtimeout then
			db[ k ] = nil
		end
	end

	-- Let's probe any players that may be connected right now.
	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) then
			t:probe( i )
		end
	end

	return t
end


--[[
	Function: newFromString

	Creates a new UCL from a string. The format of the string can be seen in this document.

	Parameters:

		str - The string to load from.

	Returns:

		The new ucl filled with values from the string. Nil and an error message otherwise.
]]
function ULib.ucl:newFromString( str )
	if not ULib.checkParam( str, "string" ) then return nil, "Incorrect parameters specified!" end
	local o = self:new()

	local lines = ULib.explode( str, "\n+" )
	lines = ULib.removeEmptyValues( lines )
	for _, line in ipairs( lines ) do
		local words = ULib.splitArgs( line )

		if table.getn( words ) >= 4 then -- Ignore invalid lines.
			local status, err = o:addUser( words[ 1 ], words[ 2 ], words[ 3 ], words[ 4 ] )
			if not status then
				ULib.debug( "Invalid line in UCL: \"" .. line .. "\", error is: \"" .. err .. "\"\n" )
			end
		else
			ULib.debug( "Invalid line in UCL: \"" .. line .. "\"\n" )
		end
	end
	
	return o
end


--[[
	Function: newFromFile

	Creates a new UCL from a file, relative to the gmod9 folder. The format of the file can be seen in this document.

	Parameters:

		file - The file to load from, relative to the gmod9 folder

	Returns:

		The new ucl filled with values from the file. Nil an and error message otherwise.
]]
function ULib.ucl:newFromFile( file )
	if not ULib.checkParam( file, "string" ) then return nil, "Incorrect parameters specified!" end
	if not _file.Exists( file ) then return nil, "Incorrect file specified!" end

	return self:newFromString( ULib.stripComments( _file.Read( file ), "//" ) )
end


--[[
	Function: obj:addUser

	Adds a new user to the UCL object "obj".

	Parameters:

		identifier - A string of the user identifier ( steamid, ip, username, clantag, etc ).
		password - A string of the password the user must use to gain access. Use "" for no password.
		access_flags - A string of the access flags to give the account.
		account_flags - A string of the account flags to give the account.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:addUser( identifier, password, access_flags, account_flags )
	if not password then password = "" end
	if not access_flags then access_flags = "" end
	if not ULib.checkParams( { identifier, account_flags }, { "string", "string" } ) or
	type( password ) ~= "string" or type( access_flags ) ~= "string" then
		return nil, "Incorrect parameters!"
	end

	local passreq = false
	if string.len( account_flags ) > 1 then
		if string.len( account_flags ) ~= 2 or not string.find( account_flags, ACCOUNT_PASSREQ ) then
			return nil, "Invalid account flags!"
		end

		account_flags = string.gsub( account_flags, ACCOUNT_PASSREQ, "" )
		passreq = true
	end

	if account_flags == ACCOUNT_PLAYERNAME then
		self:addUser_playername( identifier, password, access_flags, passreq )
	elseif account_flags == ACCOUNT_CLANTAG then
		self:addUser_clantag( identifier, password, access_flags, passreq )
	elseif account_flags == ACCOUNT_STEAMID then
		self:addUser_steamid( identifier, password, access_flags, passreq )
	elseif account_flags == ACCOUNT_IP then
		self:addUser_ip( identifier, password, access_flags, passreq )
	elseif account_flags == ACCOUNT_STEAMLOGIN then
		self:addUser_steamlogin( identifier, password, access_flags, passreq )
	else
		return nil, "Unknown account flag"
	end

	return true	
end


--[[
	Function: obj:addUser_playername

	Adds a new playername to the UCL object "obj" access list.

	Parameters:

		playername - A string of the user's playername.
		password - A string of the password the user must use to gain access. Use "" for no password.
		access_flags - A string of the access flags to give the account.
		passreq - A boolean of whether or not the password is required. The user will be disconnected on incorrect pass.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:addUser_playername( playername, password, access_flags, passreq )
	if not password then password = "" end
	if not access_flags then access_flags = "" end
	if not ULib.checkParam( playername, "string" ) or
	type( password ) ~= "string" or type( access_flags ) ~= "string" then
		return nil, "Incorrect parameters!"
	end

	if self.playernames[ playername ] then
		return nil, "playername already exists!"
	end

	self.playernames[ playername ] = { pass=password, access_flags=access_flags, passreq=passreq }
	
	if _util.PlayerByName( playername ) then
		self:probe( _util.PlayerByName( playername ) )
	end

	return true
end


--[[
	Function: obj:addUser_clantag

	Adds a new clantag (partial playername) to the UCL object "obj" access list.

	Parameters:

		clantag - A string of the clantag.
		password - A string of the password the user must use to gain access. Use "" for no password.
		access_flags - A string of the access flags to give the account.
		passreq - A boolean of whether or not the password is required. The user will be disconnected on incorrect pass.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:addUser_clantag( clantag, password, access_flags, passreq )
	if not password then password = "" end
	if not access_flags then access_flags = "" end
	if not ULib.checkParam( clantag, "string" ) or
	type( password ) ~= "string" or type( access_flags ) ~= "string" then
		return nil, "Incorrect parameters!"
	end

	if self.clantags[ clantag ] then
		return nil, "clantag already exists!"
	end

	self.clantags[ clantag ] = { pass=password, access_flags=access_flags, passreq=passreq }

	local users = ULib.getUsers( clantag, true )
	if not users then return true end -- No users, oh well!
	
	for _, userid in ipairs( users ) do
		self:probe( userid )
	end

	return true
end


--[[
	Function: obj:addUser_steamid

	Adds a new steamid to the UCL object "obj" access list.

	Parameters:

		steamid - A string of the user's steamid. You must include the "STEAM_" prefix!
		password - A string of the password the user must use to gain access. Use "" for no password.
		access_flags - A string of the access flags to give the account.
		passreq - A boolean of whether or not the password is required. The user will be disconnected on incorrect pass.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:addUser_steamid( steamid, password, access_flags, passreq )
	if not password then password = "" end
	if not access_flags then access_flags = "" end
	if not ULib.checkParam( steamid, "string" ) or
	type( password ) ~= "string" or type( access_flags ) ~= "string" then
		return nil, "Incorrect parameters!"
	end
	
	steamid = string.upper( steamid )

	if self.steamids[ steamid ] then
		return nil, "steamid already exists!"
	end

	self.steamids[ steamid ] = { pass=password, access_flags=access_flags, passreq=passreq }

	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) and _PlayerInfo( i, "networkid" ) == steamid then
			self:probe( i )
		end
	end

	return true
end


--[[
	Function: obj:addUser_ip

	Adds a new ip to the UCL object "obj" access list.

	Parameters:

		ip - A string of the user's ip.
		password - A string of the password the user must use to gain access. Use "" for no password.
		access_flags - A string of the access flags to give the account.
		passreq - A boolean of whether or not the password is required. The user will be disconnected on incorrect pass.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:addUser_ip( ip, password, access_flags, passreq )
	if not password then password = "" end
	if not access_flags then access_flags = "" end
	if not ULib.checkParam( ip, "string" ) or
	type( password ) ~= "string" or type( access_flags ) ~= "string" then
		return nil, "Incorrect parameters!"
	end

	if self.ips[ ip ] then
		return nil, "ip already exists!"
	end

	self.ips[ ip ] = { pass=password, access_flags=access_flags, passreq=passreq }

	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) and ULib.getIp( i ) == ip then
			self:probe( i )
		end
	end

	return true
end


--[[
	Function: obj:addUser_steamlogin

	Adds a new steamlogin to the UCL object "obj" access list.

	Parameters:

		steamlogin - A string of the user's steamlogin.
		password - A string of the password the user must use to gain access. Use "" for no password.
		access_flags - A string of the access flags to give the account.
		passreq - A boolean of whether or not the password is required. The user will be disconnected on incorrect pass.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:addUser_steamlogin( steamlogin, password, access_flags, passreq )
	steamlogin = string.lower( steamlogin )
	
	if not password then password = "" end
	if not access_flags then access_flags = "" end
	if not ULib.checkParam( steamlogin, "string" ) or
	type( password ) ~= "string" or type( access_flags ) ~= "string" then
		return nil, "Incorrect parameters!"
	end

	if self.steamlogins[ steamlogin ] then
		return nil, "steamlogin already exists!"
	end

	self.steamlogins[ steamlogin ] = { pass=password, access_flags=access_flags, passreq=passreq }

	for i=1, _MaxPlayers() do
		if _PlayerInfo( i, "connected" ) and string.lower( ULib.getSteamLogin( i ) ) == string.lower( steamlogin ) then
			self:probe( i )
		end
	end

	return true
end


--[[
	Function: obj:removeUser

	Removes a user from the UCL object "obj". Also removes them from the authed list.

	Parameters:

		identifier - A string of the user identifier ( steamid, ip, username, clantag, etc ).
		type - The type of account. IE, ACCOUNT_PLAYERNAME.

	Returns:

		True on success, nil and an error message otherwise.
]]
function ULib.ucl:removeUser( identifier, type )
	if not ULib.checkParams( { identifier, type }, { "string", "string" } ) then
		return nil, "Incorrect parameters!"
	end

	if string.len( type ) ~= 1 then
		return nil, "Invalid account type!"
	end

	if type == ACCOUNT_PLAYERNAME then
		if _util.PlayerByName( identifier ) then
			self.authed[ _util.PlayerByName( identifier ) ] = nil
		end
		self.playernames[ identifier ] = nil

	elseif type == ACCOUNT_CLANTAG then
		local users = ULib.getUsers( identifier, true )
		for _, userid in ipairs( users ) do
			self.authed[ userid ] = nil
		end
		self.clantags[ identifier ] = nil

	elseif type == ACCOUNT_STEAMID then
		for i=1, _MaxPlayers() do
			if _PlayerInfo( i, "connected" ) and _PlayerInfo( i, "networkid" ) == string.upper( identifier ) then
				self.authed[ i ] = nil
				break
			end
		end
		self.steamids[ identifier ] = nil

	elseif type == ACCOUNT_IP then
		for i=1, _MaxPlayers() do
			if _PlayerInfo( i, "connected" ) and ULib.getIp( userid ) == identifier then
				self.authed[ i ] = nil
			end
		end
		self.ips[ identifier ] = nil

	elseif type == ACCOUNT_STEAMLOGIN then
		for i=1, _MaxPlayers() do
			if _PlayerInfo( i, "connected" ) and string.lower( ULib.getSteamLogin( i ) ) == string.lower( identifier ) then
				self.authed[ i ] = nil
				break
			end
		end
		self.steamlogins[ identifier ] = nil

	else
		return nil, "Unknown account flag"

	end

	return true	
end


--[[
	Function: obj:probe

	Probes the user to assign access appropriately.
	*DO NOT CALL THIS DIRECTLY, UCL HANDLES IT.*

	Parameters:

		userid - The userid to probe.
		
	Revisions:

		v1.2 - Calls callback on "no" access. Changed assignment priorities around.
]]
function ULib.ucl:probe( userid )
	if self.authed[ userid ] or self.awaitingauth[ userid ] then return end -- They've already been probed

	local steamid = _PlayerInfo( userid, "networkid" )
	local name = _PlayerInfo( userid, "name" )
	local ip = ULib.getIp( userid )
	local login = string.lower( ULib.getSteamLogin( userid ) )

	local t = {} -- This will hold some important info
	t.steamlogin = login

	-- Let's figure out if they have any access
	if self.steamlogins[ login ] then
		t.access_flags = self.steamlogins[ login ].access_flags
		t.pass = self.steamlogins[ login ].pass
		t.passreq = self.steamlogins[ login ].passreq
	elseif self.steamids[ steamid ] then
		t.access_flags = self.steamids[ steamid ].access_flags
		t.pass = self.steamids[ steamid ].pass
		t.passreq = self.steamids[ steamid ].passreq
	elseif self.ips[ ip ] then
		t.access_flags = self.ips[ ip ].access_flags
		t.pass = self.ips[ ip ].pass
		t.passreq = self.ips[ ip ].passreq
	elseif self.playernames[ name ] then
		t.access_flags = self.playernames[ name ].access_flags
		t.pass = self.playernames[ name ].pass
		t.passreq = self.playernames[ name ].passreq
	elseif userid == 1 and not _IsDedicatedServer() then
		t.access_flags = ULib.LISTEN_ACCESS
		t.pass = ""
		t.passreq = false
	else
		local match
		for tag in pairs( self.clantags ) do
			if string.find( name, tag ) then
				match = tag
				break
			end
		end

		if match then
			t.access_flags = self.clantags[ match ].access_flags
			t.pass = self.clantags[ match ].pass
			t.passreq = self.clantags[ match ].passreq
		end
	end

	-- Now let's allocate access properly
	if t.access_flags then -- If they have some sort of access
		if t.pass and t.pass ~= "" then -- If there's a password, put them on the awaiting list.
			local passtimeout = tonumber( ULib.getCvar( "ulib_passtimeout" ) )
			passtimeout = passtimeout or PASSWORD_TIMEOUT -- If there's something invalid in the value, fallback on default.
			passtimeout = passtimeout * 60 -- Convert to minutes
			local info = ULib.db.ucl[ login ] -- Fetch any information they may have from the database.

			if info and info.time + passtimeout > ULib.db.uptime and info.pass == t.pass then -- If they still have an active and valid password from a previous map change
				ULib.tsay( userid, "[UCL] Access set." )
				self.authed[ userid ] = t
				self:callCallbacks( userid, t.access_flags )
				local timeago = ULib.db.uptime - info.time -- How long ago did they login in?
				ULib.addTimer( passtimeout - timeago, 1, self.passTimeout, self, userid, steamid, pass )
				return  -- They have access, we're done.
			end

			t.attempts = 0 -- TODO: add handlers for this.
			self.awaitingauth[ userid ] = t -- They don't have an active password, put them on the backburner

			if t.passreq then -- If the password is required, we have special steps.
				local passtime = tonumber( ULib.getCvar( "ulib_passtime" ) )
				passtime = passtime or PASSWORD_TIME -- If there's something invalid in the value, fallback on default.

				ULib.tsay( userid, "[UCL] You *MUST* enter your user password using the command \"_pw\" within " .. passtime .. " seconds, or be kicked." )
				ULib.addTimer( passtime, 1, self.checkAuth, self, userid )
			else
				ULib.tsay( userid, "[UCL] Please enter your user password using the command \"_pw\" to claim your admin access." )
			end

		else -- If they don't need a password, they're set!
			t.pass = nil
			ULib.tsay( userid, "[UCL] Access set." )
			self.authed[ userid ] = t
			self:callCallbacks( userid, t.access_flags ) -- Let others know they got access.
		end

	else -- They have no access, let's tell our callback.
		self:callCallbacks( userid, "" )
	end
end


--[[
	Function: obj:checkPass

	The callback for "_pw". *DO NOT CALL DIRECTLY*.
	
	Parameters:
	
		userid - The userid to check
		args - The password

	Revisions:

		v1.2 - Fixed callback
]]
function ULib.ucl:checkPass( userid, args )
	if not self.awaitingauth[ userid ] then
		if self.authed[ userid ] then
			ULib.console( userid, "[UCL] You are already authenticated." )
			return
		else
			ULib.console( userid, "[UCL] Why are you trying to enter a password when you have no access at all?" )
			return
		end
	end

	local pass = ULib.stripQuotes( args )

	if pass == self.awaitingauth[ userid ].pass then -- They got the right pass.
		self.authed[ userid ] = self.awaitingauth[ userid ]

		ULib.db.ucl[ self.awaitingauth[ userid ].steamlogin ] = { time=ULib.db.uptime, pass=self.authed[ userid ].pass } -- Log the time they logged in
		ULib.saveDb() -- Save the database now.
		
		local passtimeout = tonumber( ULib.getCvar( "ulib_passtimeout" ) )
		passtimeout = passtimeout or PASSWORD_TIMEOUT -- If there's something invalid in the value, fallback on default.
		passtimeout = passtimeout * 60 -- Convert to minutes
		ULib.addTimer( passtimeout, 1, self.passTimeout, self, userid, self.authed[ userid ].pass ) -- Add a timer for their password timeout

		self.awaitingauth[ userid ] = nil -- They are now authed, take them off awaiting.
		self.authed[ userid ].attempts = nil -- Junk cleanup
		self.authed[ userid ].pass = nil

		ULib.console( userid, "[UCL] Correct password, thank you!" )

		self:callCallbacks( userid, self.authed[ userid ].access_flags ) -- Let others know they got access
	else
		ULib.console( userid, "[UCL] Incorrect password, please try again." )
	end
end


--[[
	Function: obj:checkAuth

	The callback to make sure users enter their password within the time limit. *DO NOT CALL DIRECTLY*.
	
	Parameters:
		
		userid - The userid

	Revisions:

		v1 - Added steamid check
		v1.1 - Took off steamid parameter, uses login now.
]]
function ULib.ucl:checkAuth( userid, steamid )
	if not _PlayerInfo( userid, "connected" ) then
		self.awaitingauth[ userid ] = nil
		self.authed[ userid ] = nil
	end

	if not self.awaitingauth[ userid ] or self.awaitingauth[ userid ].steamlogin ~= ULib.getSteamLogin( userid ) then
		return -- They already authed or they're a different player
	end
	
	local name = _PlayerInfo( userid, "name" )
	ULib.console( userid, "[UCL] Password timeout, good bye." )
	ULib.kick( userid )
	ULib.tsay( _, "[UCL] " .. name .. " was kicked due to password timeout." )
end


--[[
	Function: obj:passTimeout

	The callback to time out user's passwords. *DO NOT CALL DIRECTLY*.
	
	Parameters:
		
		userid - The userid
		pass - The user's password, we need this to set them back up.
		
	Revisions:
	
		v1 - Initial
		v1.1 - Took out steamid parameter, uses login now.
]]
function ULib.ucl:passTimeout( userid, pass )
	if not IsPlayer( userid ) then
		return nil, ULib.ERR_ARGS
	end
	
	if not self.authed[ userid ] or not _PlayerInfo( userid, "connected" ) or self.authed[ userid ].steamlogin ~= ULib.getSteamLogin then
		return -- Return if they're disconnected or a different player
	end

	ULib.tsay( userid, "[UCL] Your password has timed out, please enter your user password using the command \"_pw\" to reclaim your admin access." )
	self.awaitingauth[ userid ] = self.authed[ userid ]
	self.authed[ userid ] = nil

	self.awaitingauth[ userid ].pass = pass -- Set them back up to get auth
	self.awaitingauth[ userid ].attempts = 0
end

--[[
	Function: obj:callCallbacks

	*Internal function, DO NOT CALL DIRECTLY.* This function calls access callbacks
	
	Parameters:
		
		userid - The userid who got access.
		access - The access string.
		
	Revisions:
	
		v1.1 - Initial
		v1.2 - Made it harder to break callbacks
]]
function ULib.ucl:callCallbacks( userid, access )
	ULib.ucl.tempTable = self.callbacks -- Have to make it global so runstring can get to it.

	for key, fn in ipairs( self.callbacks ) do
		_RunString( "ULib.ucl.tempTable[ " .. key .. " ]( " .. userid .. ", " .. string.format( "%q", access ) .. " )" )
	end

	ULib.ucl.tempTable = nil
end