--[[
	Title: ULib Plugin

	This is the lua handler of the ULib plugin. None of these functions will work without the plugin (which only works on dedicated servers).

	Revisions:

		v1.2 - Initial
]]

--[[
	Function: pluginEnabled

	Checks if the ULib plugin is enabled on this server.

	Returns:

		True if it's enabled, false otherwise.
		
	Revisions:

		v1.2 - Initial
]]
function ULib.pluginEnabled()
	return _GetConVar_Bool( "ulib_plugin_enabled" )
end

--[[
	Function: cexec

	Executes a command on a client.

	Parameters:

		userid - The user to execute on.
		args - The command to pass to their console.

	Returns:

		True on success, nil and error otherwise.
		
	Revisions:

		v1.2 - Initial
]]
function ULib.cexec( userid, args )
	if not ULib.pluginEnabled() then
		return nil, "Plugin is not enabled!"
	end

	if not ULib.checkParams( { userid, args }, { "number", "string" } ) then
		return nil, ULib.ERR_ARGS
	end
	
	if not _PlayerInfo( userid, "connected" ) then
		return nil, "Tried to execute on an unconnected player!"
	end

	_ServerCommand( "ulib_plugin_cexec " .. userid .. " " .. args .. "\n" )
	return true
end

--[[
	Variable: startTime

	This variable, if the ULib plugin is enabled, holds the time in seconds from midnight the server started.
]]
ULib.startTime = nil

--[[
	Variable: timeZone

	This variable, if the ULib plugin is enabled, holds the timezone of the server.
]]
ULib.timeZone = nil

--[[
	Function: setTime

	*DO NOT CALL DIRECTLY*. This function receives the time callback
	
	Parameters:
		
		h - Hours.
		m - Minutes.
		s - Seconds.
		tz - Timezone.

	Revisions:

		v1.2 - Initial
]]
function ULib.setTime( h, m, s, tz )
	local secs = 0
	secs = secs + h * 3600
	secs = secs + m * 60
	secs = secs + s

	local newStartTime = secs - math.floor( _CurTime() )
	if ULib.startTime and newStartTime < ULib.startTime then -- Must be a new day
		ULib.startTime = newStartTime
		ULib.syncDate()
	end
	ULib.startTime = newStartTime
	ULib.timeZone = tz
end

--[[
	Function: getTime

	Gets the real world time.

	Returns:

		Time in seconds from midnight.

	Revisions:

		v1.2 - Initial
]]
function ULib.getTime()
	if not ULib.pluginEnabled() then
		return nil, "Plugin is not enabled!"
	end

	if ULib.startTime + _CurTime() >= 24 * 3600 then -- Uh oh! New day!
		ULib.startTime = math.mod( ULib.startTime, 24 * 3600 ) -- Temp handle of time till we resync.
		ULib.syncTime()
		ULib.syncDate()
	end

	return ULib.startTime + _CurTime()
end

--[[
	Function: syncTime

	Synchronizes ULib's real world time with the system clock. You should never have to call this, as ULib handles it.

	Returns:

		True on success, nil and error otherwise.

	Revisions:

		v1.2 - Initial
]]
function ULib.syncTime()
	if not ULib.pluginEnabled() then
		return nil, "Plugin is not enabled!"
	end

	_ServerCommand( "ulib_plugin_gettime ULib.setTime\n" )
	return true
end


--[[
	Variable: date

	This variable, if the ULib plugin is enabled, holds the date.
]]
ULib.date = nil

--[[
	Function: setDate

	*DO NOT CALL DIRECTLY*. This function receives the date callback

	Parameters:

		m - Month.
		d - Day.
		y - Year.
		wd - Weekday.

	Revisions:

		v1.2 - Initial
]]
function ULib.setDate( m, d, y, wd )
	ULib.date = { m=m, d=d, y=y, wd=wd }
end

--[[
	Function: getDate

	Gets the real world date.

	Returns:

		Month, day, year, weekday. Returns nil and error otherwise.

	Revisions:

		v1.2 - Initial
]]
function ULib.getDate()
	if not ULib.pluginEnabled() then
		return nil, "Plugin is not enabled!"
	end

	return ULib.date.m, ULib.date.d, ULib.date.y, ULib.date.wd
end

--[[
	Function: syncDate

	Synchronizes ULib's real world date with the system clock. You should never have to call this, as ULib handles it.

	Returns:

		True on success, nil and error otherwise.

	Revisions:

		v1.2 - Initial
]]
function ULib.syncDate()
	if not ULib.pluginEnabled() then
		return nil, "Plugin is not enabled!"
	end

	_ServerCommand( "ulib_plugin_getdate ULib.setDate\n" )
	return true
end

if ULib.pluginEnabled() then
	ULib.syncTime()
	ULib.syncDate()
	ULib.addTimer( 10, 0, ULib.syncTime ) -- Resync every 10 seconds. (_CurTime freezes during prop precache)
end