--[[
	Title: Color

	Contains a color chart and various color helpers
]]


--[[
	Function: getColor

	Retrieves a color.

	Parameters:

		name - The name of the color, as a string

	Returns:

		A table with values r, g, and b, corresponding to red, green, and blue respectively. Nil and error message otherwise.
]]
function ULib.getColor( name )
	if not ULib.checkParam( name, "string" ) then return nil, ULib.ERR_ARGS end
	
	name = string.lower( name )
	if not ULib.colorIndex[ name ] then return nil, "No such color" end
	return ULib.colorIndex[ name ]
end


--[[
	Function: getColorString

	Retrieves a color as a string.

	Parameters:

		name - The name of the color, as a string

	Returns:

		A keyvalue-friendly string. ( "r g b" ). Nil and error message otherwise.
]]
function ULib.getColorString( name )
	local rgb, err = ULib.getColor( name )
	if not rgb then return nil, err end

	if rgb and rgb.r and rgb.g and rgb.b then
		return rgb.r .. " " .. rgb.g .. " " .. rgb.b
	end
end


--[[
	Function: applyColorToEnt

	Applys the specified color to an entity

	Parameters:

		entid - The entity index location of the entity to apply the color to
		name - The name of the color, as a string
		
	Returns:
		On error only, nil and error message.
]]
function ULib.applyColorToEnt( entid, color )
	local colorStr, err = ULib.getColorString( color )
	if not colorStr then return nil, err end

	_EntFire( entid, "color", colorStr, 0 )
end


--[[
	Table: colorIndex

	Contains all color definitions

	Colors:
]]
ULib.colorIndex = {}

ULib.colorIndex.brightred = { r = 255, g = 0, b = 0 }
ULib.colorIndex.brightgreen = { r = 0, g = 255, b = 0 }
ULib.colorIndex.green = { r = 0, g = 128, b = 0 }
ULib.colorIndex.red = { r = 192, g = 0, b = 0 }
ULib.colorIndex.blue = { r = 0, g = 0, b = 255 }
ULib.colorIndex.magenta = { r = 255, g = 0, b = 255 }
ULib.colorIndex.yellow = { r = 255, g = 255, b = 0 }
ULib.colorIndex.cyan = { r = 0, g = 255, b = 255 }
ULib.colorIndex.white = { r = 255, g = 255, b = 255 }
ULib.colorIndex.grey = { r = 128, g = 128, b = 128 }
ULib.colorIndex.gray = { r = 128, g = 128, b = 128 }
ULib.colorIndex.black = { r = 0, g = 0, b = 0 }
ULib.colorIndex.brown = { r = 128, g = 64, b = 0 }
ULib.colorIndex.violet = { r = 128, g = 0, b = 128 }
ULib.colorIndex.purple = { r = 128, g = 0, b = 255 }
ULib.colorIndex.pink = { r = 255, g = 0, b = 128 }
ULib.colorIndex.lightpink = { r = 255, g = 128, b = 192 }
ULib.colorIndex.navy = { r = 0, g = 0, b = 128 }
ULib.colorIndex.teal = { r = 0, g = 128, b = 192 }
ULib.colorIndex.orange = { r = 255, g = 128, b = 0 }
ULib.colorIndex.khaki = { r = 128, g = 128, b = 64 }
ULib.colorIndex.maroon = { r = 128, g = 0, b = 0 }
ULib.colorIndex.forest = { r = 0, g = 128, b = 0 }
ULib.colorIndex.sky = { r = 128, g = 128, b = 255 }
ULib.colorIndex.hotpink = { r = 255, g = 128, b = 255 }
ULib.colorIndex.deepviolet = { r = 64, g = 0, b = 64 }
ULib.colorIndex.darkgrey = { r = 64, g = 64, b = 64 }
ULib.colorIndex.darkgray = { r = 64, g = 64, b = 64 }
ULib.colorIndex.lightgrey = { r = 192, g = 192, b = 192 }
ULib.colorIndex.lightgray = { r = 192, g = 192, b = 192 }
ULib.colorIndex.peach = { r = 255, g = 128, b = 128 }
ULib.colorIndex.lightbrown = { r = 128, g = 128, b = 0 }
ULib.colorIndex.tan = { r = 128, g = 128, b = 64 }
ULib.colorIndex.puke = { r = 0, g = 255, b = 128 }
ULib.colorIndex.lightyellow = { r = 255, g = 255, b = 128 }
ULib.colorIndex.citrus = { r = 255, g = 128, b = 64 }
ULib.colorIndex.darkbrown = { r = 64, g = 64, b = 16 }
ULib.colorIndex.gold = { r = 144, g = 144, b = 16 }