--[[
	Title: Entity Helpers
]]


--[[
	Function: addOutput

	Adds an output to an entity event

	Parameters:

		entid - The entity to add an output to
		output - String representation of the output to apply this to
		entname - Either the entity index location or name of the entity to apply this to.
		dothis - What to fire on the entity
		params - Parameters to pass on the entity
		wait - How long to wait after the output is called before this is run
		times - How many times the output can fire before it stops working.

	Example:
	
		ULib.addOutput( 1, "OnHealthChanged", "!activator", "ignite", _, 1, -1 )  -- Make it so that when someone shoots player 1, they are ignited
		
	Revisions:
	
		v1.1 - Fixed parameters
]]
function ULib.addOutput( entid, output, entname, dothis, params, wait, times )
	params = params or 1
	wait = wait or 0
	times = times or -1
	if type( entname ) == "number" then
		entname = ULib.generateEntName( entname )
	end
	_EntFire( entid, "addoutput", output .. " " .. entname .. ":" .. dothis .. ":" .. params .. ":" .. wait .. ":" .. times, 0 )
end


--[[
	Function: generateEntName

	Generates or returns a name for an entity

	Parameters:

		entid - The entity to generate a name for.

	Returns:

		Either the entity's current name, or a new name for the entity if it doesn't have one. Nil and error otherwise.
		
	Revisions:
	
		v1.1 - Made it more reliable
]]
function ULib.generateEntName( entid )
	if not _EntExists( entid ) then return nil, "No such ent!" end

	if _EntGetName( entid ) ~= "" and _EntGetName( entid ) ~= _EntGetType( entid ) then
		return _EntGetName( entid )
	else
		_EntSetName( entid, "ent_" .. entid )
		return "ent_" .. entid
	end
end


--[[
	Function: createRunFunction

	Creates and spawns a working gmod_runfunction entity. For interface between entitys and Lua.

	Parameters:

		functionname - A string that is the name of the function to call.

	Returns:

		The entity index location of the runfunction entity.
]]
function ULib.createRunFunction ( functionname )
	local temp = _EntCreate( "gmod_runfunction" )
	_EntSetKeyValue( temp, "functionname", functionname )
	_EntSpawn( temp )
	return temp
end


--[[
	Function: fireRunFunction

	Runs the function of a gmod_runfunction

	Parameters:

		entid - The entity index location of the gmod_runfunction

]]
function ULib.fireRunFunction( entid )
	if _EntGetType( entid ) == "gmod_runfunction" then
		_EntFire( entid, "runscript", "", 0 )
	end
end


--[[
	Function: fireRunFunctionOnOutput

	Runs the function of a gmod_runfunction on the specified output

	Parameters:

		entid - The entity index location of the gmod_runfunction
		fireid - The entity index location of the entity to apply the output to
		output

]]
function ULib.fireRunFunctionOnOutput( entid, fireid, output )
	if _EntGetType( entid ) == "gmod_runfunction" then
		local runfunction = ULib.generateName( entid )
		ULib.addOutput( fireid, output, runfunction, "runscript" )
	end
end


--[[
	Function: removeAllEnts

	Removes all entitys with the specified name and classname

	Parameters:

		name - The name to remove

]]
function ULib.removeAllEnts( name )
	local names = _EntitiesFindByName( name )
	local class = _EntitiesFindByClass( name )
	for _, entid in names do
		if _EntExists( entid ) then
			_EntRemove( entid )
		end
	end
	for _, entid in class do
		if _EntExists( entid ) then
			_EntRemove( entid )
		end
	end
end


--[[
	Function: makeProp

	Create a prop

	Parameters:

		model - The model of the prop
		pos - The position of the prop
		ang - The angle of the prop
		
	Returns:
	
		The entity index location of the prop.

]]
function ULib.makeProp( model, pos, ang )
	local temp = _EntCreate( "prop_physics" )
	_EntPrecacheModel( model )
	_EntSetPos( temp, pos )
	_EntSetAng( temp, ang )
	_EntSetModel( temp, model )
	_EntSpawn( temp )
	return temp
end


--[[
	Function: makeRagdoll

	Create a ragdoll

	Parameters:

		model - The model of the ragdoll
		pos - The position of the ragdoll
		
	Returns:
	
		The entity index location of the ragdoll.

]]
function ULib.makeRagdoll( model, pos )
	local temp = _EntCreate( "prop_ragdoll" )
	_EntPrecacheModel( model )
	_EntSetKeyValue( temp, "origin", vecString( pos ) )
	_EntSetModel( temp, model )
	_EntSpawn( temp )
	return temp
end

--[[
	Table: cballTimes

	*INTERNAL USE ONLY*. This helps us determine if a cball has already been registered.
	
	Revisions:
	
		v1.1 - Initial
]]
local cballTimes = {}

--[[
	Function: registerCball

	*DO NOT CALL DIRECTLY*. This function supports <makeCball>.

	Parameters:

        	callback - The function to call when the ball is made.
        	pos - The position the spawner was made at, used to help track down the cball.
		... - *(Optional)* Any number of extra parameters you wanted passed to the callback.
		
	Revisions:
	
		v1.1 - Initial
]]
local function registerCball( callback, pos, ... )
	local ents = _EntitiesFindByClass( "prop_combine_ball" )
	--local num = table.getn( ents ) -- Should be at the end of the list, since it's the newest.
	--if num < 1 then return end -- ERROR
	
	local allow = 3.1 -- Allowed difference in x and y
	local allow_z = 12 -- Allowed difference in z
	local deny_time = 0.1
	local entid

	for i=table.getn( ents ), 1, -1 do
		entid = ents[ i ]
		local entpos = _EntGetPos( entid )
		if _EntGetOwner( entid ) and -- HACK: Player's cball's don't have an owner, this is just another filter for us.
		   math.abs( entpos.x - pos.x ) < allow and
		   math.abs( entpos.y - pos.y ) < allow and
		   math.abs( entpos.z - pos.z ) < allow_z and
		   (not cballTimes[ entid ] or cballTimes[ entid ] + deny_time < _CurTime()) -- Filter out already registered cballs
		   then
			cballTimes[ entid ] = _CurTime()
			break
		end
		
		if i == 1 then -- We weren't able to find it.
			_Msg( "Error! Cball not found!\n" )
			return
		end
	end

	callback( entid, unpack( arg ) )
end


--[[
	Function: makeCball

	Makes a combine ball. Note that due to limitations, the ball is not spawned instantly. 
	Please specify a callback which will be called when the ball is made. Parameters will be
	the entid of the cball and whatever extra parameters you pass to this function.

	Parameters:

        	callback - The function to call when the ball is made.
		pos - The position of the cball.
		size - *(Optional, defaults to 15)* The radius of the cball.
		... - *(Optional)* Any number of extra parameters you wanted passed to the callback.

	Returns:

		True on success, nil and error otherwise.
		
	Revisions:
	
		v1.1 - Initial		
]]
function ULib.makeCball( callback, pos, size, ... )
	if not ULib.checkParam( callback, "function" ) or not pos then return nil, ULib.ERR_ARGS end
	size = size or 15

	local entid = _EntCreate( "func_combine_ball_spawner" )
	if entid == 0 then return nil, "Could not make the entity!" end

	_EntSetModel( entid, "models/weapons/w_bugbait.mdl" ) -- This will fool the engine into thinking this is a proper brush ent, but it will generate an error.
	_EntSetKeyValue( entid, "maxspeed", "0" )
	_EntSetKeyValue( entid, "ballradius", tostring( size ) )
	_EntSetKeyValue( entid, "ballrespawntime", "0.01" )
	_EntSetKeyValue( entid, "ballcount", "1" )
	_EntSetPos( entid, pos )
	_EntSpawn( entid )
	_EntFire( entid, "kill", "", 0.1 ) -- We won't need the spawner anymore.

	ULib.addTimer( 0.1, 1, registerCball, callback, pos, unpack( arg ) )
	return true
end

--[[
	Function: keepUpright

	Adds a phys_keepupright to an entity. Does what you'd think.

	Parameters:

        	entid - The entid to keep upright
        	limit - *(Optional, defaults to 10)* How many degrees per second it will compensate for.

	Returns:

		entid on success, nil and error otherwise.
		
	Revisions:

		v1.1 - Initial
		v1.2 - Added angle argument
]]
function ULib.keepUpright( entid, limit, angles )
	if not ULib.checkParam( entid, "number" ) then return nil, ULib.ERR_ARGS end
	if not _EntExists( entid ) then return nil, "No such ent." end

	limit = limit or 10
	angles = angles or vector3( 0, 0, 0 )

	local uprightid = _EntCreate( "phys_keepupright" )
	_EntSetKeyValue( uprightid, "attach1", ULib.generateEntName( entid ) )
	_EntSetKeyValue( uprightid, "angularlimit", tostring( limit ) )
	_EntSetKeyValue( uprightid, "angles", tostring( angles ) )
	_EntSpawn( uprightid )
	_EntActivate( uprightid )

	return uprightid
end



--[[
	Function: callOnDelete

	Calls a function when a specific ent is deleted. Note that unlike just simply adding an output for OnBreak, this will catch all deletion methods.

	Parameters:

		entid - The entid you're watching for a delete with.
		func - The function to call on deletion.
		... - *(Optional)* Specify as many parameters as you'd like that will be passed to func.
		
	Example:
	
		ULib.callOnDelete( _EntGetByName( "doda" ), PlayerFreezeAll, true )
		This will freeze all players when the prop by the name of "doda" is deleted.
]]
function ULib.callOnDelete( entid, func, ... )
	_G[ "gEntDelete" .. entid ] = function( timed )
		if not timed then ULib.addTimer( 0.01, 1, _G[ "gEntDelete" .. entid ], true ) end -- We use timed because deletions won't show up for a second.
		if _EntExists( entid ) then return end -- Not gone yet

		func( unpack( arg ) )
		_G[ "gEntDelete" .. entid ] = nil
	end

	ULib.addOutput( entid, "OnHealthChanged", "!self", "runfunction", "gEntDelete" .. entid, 0, -1 )
end