--[[
	Title: Messages

	Handles messaging like logging, debug, etc.
]]


--[[
	Function: tsay

	Prints a message in talk say as well as in the user's consoles.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		msg - The message to print.
]]
function ULib.tsay( userid, msg )
	userid = userid or 0
	if userid ~= 0 then
		_PrintMessage( userid, HUD_PRINTTALK, msg .. "\n" )
		_PrintMessage( userid, HUD_PRINTCONSOLE, msg .. "\n" )
	else
		_PrintMessageAll( HUD_PRINTTALK, msg .. "\n" )
		_PrintMessageAll( HUD_PRINTCONSOLE, msg .. "\n" )
	end
end


--[[
	Function: csay

	Prints a message in center of the screen as well as in the user's consoles.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		msg - The message to print.
		r - *(Optional)* The amount of red to use for the text.
		g - *(Optional)* The amount of green to use for the text.
		b - *(Optional)* The amount of blue to use for the text.
		a - *(Optional)* The alpha to use for the text.
		length - *(Optional)* The amount of time to show the text.
]]
function ULib.csay( userid, msg, r, g, b, a, length )
	userid = userid or 0
	length = length or 5
	if not userid or type( userid ) ~= "number" then
		userid = 0
	end

	ULib.sendText( userid, 369, -1, 0.25,  msg .. "\n", length, r, g, b, a, "TargetID" )

	if userid == 0 then
		_PrintMessageAll( HUD_PRINTCONSOLE, msg .. "\n" )
	else
		_PrintMessage( userid, HUD_PRINTCONSOLE, msg .. "\n" )
	end
end


--[[
	Function: console

	Prints a message in the user's consoles.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		msg - The message to print.
]]
function ULib.console( userid, msg )
	userid = userid or 0
	if userid ~= 0 then
		_PrintMessage( userid, HUD_PRINTCONSOLE, msg .. "\n" )	
	else
		_PrintMessageAll( HUD_PRINTCONSOLE, msg .. "\n" )
	end
end


--[[
	Function: sendText

	Sends text to the user's screen.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		key - The key to use for the text.
		posx - The position x to use.
		posy - The position y to use.
		msg - The message to print.
		duration - The time to display the text
		r - *(Optional, defaults to 255)* The amount of red to use.
		g - *(Optional, defaults to 255)* The amount of green to use.
		b - *(Optional, defaults to 255)* The amount of blue to use.
		a - *(Optional, defaults to 255)* The alpha to use.
		font - *(Optional, defaults to "Default")* The font to use.
		fadein - *(Optional, defaults to 0.5)* The time it will take for the text to fade in.
		fadeout - *(Optional, defaults to 0.5)* The time it will take for the text to fade out.
		delay - *(Optional, defaults to 0)* The delay between the time this is called and fadein begins.
]]
function ULib.sendText( userid, key, posx, posy, msg, duration, r, g, b, a, font, fadein, fadeout, delay )
	userid = userid or 0
	r = r or 255	
	g = g or 255
	b = b or 255
	a = a or 255
	font = font or "Default"
	fadein = fadein or 0.5
	fadeout = fadeout or 0.5
	delay = delay or 0

	_GModText_Start( font )
		_GModText_SetPos( posx, posy )
		_GModText_SetColor( r, g, b, a )
		_GModText_SetTime( duration, fadein, fadeout )
		_GModText_SetText( msg )
		_GModText_SetDelay( delay )
		_GModText_AllowOffscreen( false )
	_GModText_Send( userid, key )
end


--[[
	Function: sendTextAnimate

	Only animates a text.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		key - The key to use for the text.
		posx2 - The position x to end up in.
		posy2 - The position y to end up in.
		time - The time for the animation to take
		ease - *(Optional, defaults to 0.5)* The ease to use for the animation, 0.5 = no ease.
		font - *(Optional, defaults to "Default")* The font to use.
		delay2 - *(Optional, defaultsto 0)* The delay between the initial text and the animation.
]]
function ULib.sendTextAnimate( userid, key, posx2, posy2, time, ease, font, delay2 )
	userid = userid or 0
	ease = ease or 0.5
	font = font or "Default"
	delay2 = delay2 or 0

	_GModText_Start( font )
		_GModText_SetPos( posx2, posy2 )
		_GModText_SetDelay( delay2 )
	_GModText_SendAnimate( userid, key, time, ease )
end



--[[
	Function: sendTextAndAnimate

	Sends text and animates it on the user's screen.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		key - The key to use for the text.
		posx - The position x to use.
		posy - The position y to use.
		posx2 - The position x to end up in.
		posy2 - The position y to end up in.
		msg - The message to print.
		duration - The time to display the text
		time - The time for the animation to take
		ease - *(Optional, defaults to 0.5)* The ease to use for the animation, 0.5 = no ease.
		r - *(Optional, defaults to 255)* The amount of red to use.
		g - *(Optional, defaults to 255)* The amount of green to use.
		b - *(Optional, defaults to 255)* The amount of blue to use.
		a - *(Optional, defaults to 255)* The alpha to use.
		font - *(Optional, defaults to "Default")* The font to use.
		fadein - *(Optional, defaults to 0.5)* The time it will take for the text to fade in.
		fadeout - *(Optional, defaults to 0.5)* The time it will take for the text to fade out.
		delay - *(Optional, defaults to 0)* The delay between the time this is called and fadein begins.
		delay2 - *(Optional, defaultsto 0)* The delay between the initial text and the animation.
]]
function ULib.sendTextAndAnimate( userid, key, posx, posy, posx2, posy2, msg, duration, time, ease, r, g, b, a, font, fadein, fadeout, delay, delay2 )
	ULib.sendText( userid, key, posx, posy, msg, duration, r, g, b, a, font, fadein, fadeout, delay )
	ULib.sendTextAnimate( userid, key, posx2, posy2, time, ease, font, delay2 )
end


--[[
	Function: sendRect

	Sends a rect to the user's screen.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		key - The key to use for the rect.
		posx - The position x to use.
		posy - The position y to use.
		width - The width to use.
		height - The height to use.
		duration - The time to display the rect.
		r - *(Optional, defaults to 255)* The amount of red to use.
		g - *(Optional, defaults to 255)* The amount of green to use.
		b - *(Optional, defaults to 255)* The amount of blue to use.
		a - *(Optional, defaults to 255)* The alpha to use.
		material - *(Optional, defaults to "gmod/white)* The material to use. Note, if you're using a custom material, you must add it to the download list.
		fadein - *(Optional, defaults to 0.5)* The time it will take for the rect to fade in.
		fadeout - *(Optional, defaults to 0.5)* The time it will take for the rect to fade out.
		delay - *(Optional, defaults to 0)* The delay between the time this is called and fadein begins.
]]
function ULib.sendRect( userid, key, posx, posy, width, height, duration, r, g, b, a, material, fadein, fadeout, delay )
	userid = userid or 0
	r = r or 255	
	g = g or 255
	b = b or 255
	a = a or 255
	material = material or "gmod/white"
	fadein = fadein or 0.5
	fadeout = fadeout or 0.5
	delay = delay or 0

	_GModRect_Start( material )
		_GModRect_SetPos( posx, posy, width, height )
		_GModRect_SetColor( r, g, b, a )
		_GModRect_SetTime( duration, fadein, fadeout )
		_GModRect_SetDelay( delay )
	_GModRect_Send( userid, key )
end

--[[
	Function: sendRectAnimate

	Only animates a rect.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		key - The key to use for the rect.
		posx2 - The position x to end up in.
		posy2 - The position y to end up in.
		width2 - The width to end up as.
		height2 - The height to end up as.
		time - The time for the animation to take
		ease - *(Optional, defaults to 0.5)* The ease to use for the animation, 0.5 = no ease.
		material - *(Optional, defaults to "gmod/white)* The material to use. Note, if you're using a custom material, you must add it to the download list.
		delay2 - *(Optional, defaultsto 0)* The delay between the initial rect and the animation.
]]
function ULib.sendRectAnimate( userid, key, posx2, posy2, width2, height2, time, ease, material, delay2 )
	userid = userid or 0
	ease = ease or 0.5
	material = material or "gmod/white"
	delay2 = delay2 or 0

	_GModText_Start( font )
		_GModText_SetPos( posx2, posy2, width2, height2 )
		_GModText_SetDelay( delay2 )
	_GModText_SendAnimate( userid, key, time, ease )
end



--[[
	Function: sendRectAndAnimate

	Sends a rect and animates it on the user's screen.

	Parameters:

		userid - The userid to print to, set to 0 or nil to send to everyone.
		key - The key to use for the rect.
		posx - The position x to use.
		posy - The position y to use.
		width - The width to use.
		height - The height to use.
		posx2 - The position x to end up in.
		posy2 - The position y to end up in.
		width2 - The width to end up as.
		height2 - The height to end up as.
		duration - The time to display the rect.
		time - The time for the animation to take
		ease - *(Optional, defaults to 0.5)* The ease to use for the animation, 0.5 = no ease.
		r - *(Optional, defaults to 255)* The amount of red to use.
		g - *(Optional, defaults to 255)* The amount of green to use.
		b - *(Optional, defaults to 255)* The amount of blue to use.
		a - *(Optional, defaults to 255)* The alpha to use.
		material - *(Optional, defaults to "gmod/white)* The material to use. Note, if you're using a custom material, you must add it to the download list.
		fadein - *(Optional, defaults to 0.5)* The time it will take for the rect to fade in.
		fadeout - *(Optional, defaults to 0.5)* The time it will take for the rect to fade out.
		delay - *(Optional, defaults to 0)* The delay between the time this is called and fadein begins.
		delay2 - *(Optional, defaultsto 0)* The delay between the initial rect and the animation.
		
	Revisions:
	
		v1.2 - Fixed. Do not use this function in pre v1.2.
]]
function ULib.sendRectAndAnimate( userid, key, posx, posy, width, height, posx2, posy2, width2, height2, duration, time, ease, r, g, b, a, material, fadein, fadeout, delay, delay2 )
	ULib.sendRect( userid, key, posx, posy, width, height, duration, r, g, b, a, material, fadein, fadeout, delay )
	ULib.sendRectAnimate( userid, key, posx2, posy2, width2, height2, time, ease, material, delay2 )
end


--[[
	Function: debug

	Logs the parameter given to console as a debug message using <log()>

	Parameters:

		s - The string to use as the debug message
		level - The debug level, see <DEBUG_LEVEL>

	See Also:

		<log()>

		<DEBUG_MODE>
]]
function ULib.debug( s, level )
	s = s or "No parameter passed to ULib.debug!"
	level = level or ULib.ERR
	
	if level >= ULib.DEBUG_LEVEL then
		ULib.log( "[ULIB] DEBUG: " .. s )
	end
end


--[[
	Function: debugFunctionCall

	Prints a function call, very useful for debugging.
	Note that this prints on debug level 2. See <DEBUG_MODE>

	Parameters:

		name - The name of the function called
		... - all arguments to the function
]]
function ULib.debugFunctionCall( name, ... )
	local params = ""

	for i=1, table.getn( arg ) do
		local value = arg[ i ]
		if type( value ) == "string" then
			value = string.format( "%q", value )
		else
			value = tostring( value )
		end
		params = params .. "[PARAMETER " .. i .. "]: Type=" .. type( arg[ i ] ) .. "\tValue=(" .. value .. ")\n"
	end
	ULib.debug( "Function '" .. name .. "' called. Parameters:\n" .. params, ULib.DEBUG )
end


--[[
	Function: log

	Outputs the parameter given to console and writes it in the log if <LOG> is true.
	This function adds the newline itself.

	See Also:

		<LOG>

		<LOG_FILE>
		
	Revisions:
	
		v1.1 - Fixed
]]
function ULib.log( s )
	if not s then ULib.debug( "No value passed to ULib.log!", 1 ) return end

	_Msg( s .. "\n" )
	if ULib.LOG then
		_file.Write( ULib.LOG_FILE, _file.Read( ULib.LOG_FILE ) .. s .. ULib.NEWLINE )
	end
end


--[[
	Function: print

	Prints a message from the given arguments. This acts *exactly* like lua's print().
	The key feature is that it will print *all arguments given to the function.*
	This is great for printing the results of a function call, etc.
]]
function ULib.print( ... )
	local msg = ""

	for i=1, table.getn( arg ) do
		msg = msg .. tostring( arg[ i ] ) .. ULib.SPACE_METHOD
	end
	_Msg( msg .. "\n" )
end