--[[
	Title: Say

	Handles say hooks

	Everything in this file started in v1.
]]

--[[
	Table: sayHooks

	This table holds say hook functions.
]]
ULib.sayHooks = { {}, {}, {}, {} }
local gSayHooks = ULib.sayHooks -- Because I'm lazy and don't want to replace these.

--[[
	Function: eventPlayerSay

	Handles the player say callback, also handles custom hooks.

	Parameters:

		userid - The userid saying something
		strText - What they're saying
		bTeam - The team they're on

	Returns:

		The player say string after processing.
		
	Revisions:
	
		v1.2 - Won't stop processing due to a broken function.
]]
function ULib.eventPlayerSay( userid, strText, bTeam )
	for key, fn in ipairs( gSayHooks[ 1 ] ) do
		_RunString( "ULib.sayHooks[ 1 ][ " .. key .. " ]( " .. userid .. ", " .. string.format("%q", strText) .. ", " .. tostring(bTeam) .. " )" ) -- We purposely ignore the return value.
	end

	for key, fn in ipairs( gSayHooks[ 2 ] ) do
		_RunString( "ULib.strReturn = ULib.sayHooks[ 2 ][ " .. key .. " ]( " .. userid .. ", " .. string.format("%q", strText) .. ", " .. tostring(bTeam) .. " )" )
		strText = ULib.strReturn
		ULib.strReturn = nil
		if not strText or strText == "" then return "" end -- Don't continue, their message is killed.
	end


	for key, fn in ipairs( gSayHooks[ 3 ] ) do
		_RunString( "ULib.strReturn = ULib.sayHooks[ 3 ][ " .. key .. " ]( " .. userid .. ", " .. string.format("%q", strText) .. ", " .. tostring(bTeam) .. " )" )
		strText = ULib.strReturn
		ULib.strReturn = nil
		if not strText or strText == "" then return "" end -- Don't continue, their message is killed.
	end


	for key, fn in ipairs( gSayHooks[ 4 ] ) do
		_RunString( "ULib.sayHooks[ 4 ][ " .. key .. " ]( " .. userid .. ", " .. string.format("%q", strText) .. ", " .. tostring(bTeam) .. " )" ) -- We purposely ignore the return value.
	end
	
	return strText
end
eventPlayerSay = ULib.eventPlayerSay


--[[
	Function: checkSay

	This function makes sure ULib's eventPlayerSay always remains. If it finds a conflicting function, it adds it to ULib's hooks.
]]
local function checkSay()
	if eventPlayerSay ~= ULib.eventPlayerSay then
		if not ULib.isInTable( gSayHooks, eventPlayerSay, true ) then 
			ULib.hookSay( eventPlayerSay, 3 )
		end
		eventPlayerSay = ULib.eventPlayerSay -- Better to save as many hooks as possible then let another script override them.
	end
end
ULib.addTimer( 10, 0, checkSay )


--[[
	Function: hookSay

	This function adds a say hook.

	Parameters:
		
		fn - The function to callback on a say
		level - The priority to give this callback. You can pass 1, 2, 3, or 4. 

		1 is for *PASSIVE CALLBACKS ONLY*, these functions will not be allowed to change the say string. Level 1 is useful for logging functions.

		2 is for high priority callbacks, functions that need to edit the string first. Level 2 is useful for things like command callbacks.

		3 is for low priority callbacks, functions that don't need to see the string in pristine form. Level 3 is useful for things like swear filters.

		4 is for lowest priority *PASSIVE CALLBACKS ONLY*, these functions will not be allowed to change the say string. Level 4 is useful for things like an IRC echo script. Note that this function is not guaranteed to be called if another function returns "".
]]
function ULib.hookSay( fn, level )
	if not ULib.checkParams( { fn, level }, { "function", "number" } ) then return nil, ULib.ERR_ARGS end
	if level < 1 or level > 4 then return nil, "Out of range level" end
	if ULib.isInTable( gSayHooks, fn, true ) then return nil, "Function already hooked!" end

	table.insert( gSayHooks[ level ], fn )
end


--[[
	Function: unHookSay

	This function unhooks say hook.

	Parameters:
		
		fn - The function callback to unhook
]]
function ULib.unHookSay( fn )
	if not ULib.checkParam( fn, "function" ) then return nil, ULib.ERR_ARGS end
	if not ULib.isInTable( gSayHooks, fn, true ) then return nil, "Function not hooked!" end

	for level in ipairs( gSayHooks ) do
		for i, v in ipairs( gSayHooks[ level ] ) do
			if v == fn then table.remove( gSayHooks[ level ], i ) end -- Remove it
		end
	end
end


--[[
	Table: sayCmds

	This holds the say commands, note that it is not local.
]]
ULib.sayCmds = {}


--[[
	Function: sayCmdCheck
	
	Say callback which will check to see if there's a say command being used. *DO NOT CALL DIRECTLY*

	Parameters:
		
		say_cmd - A command string for says. IE: "!kick", then when someone says "!kick", it'll call the callback.
		fn_call - The function to call when the command's called.
		cmd_help - *(Optional)* A help string for using the command.
		hide_say - *(Optional, defaults to false)* If true, will hide the chat message. Use this if you don't want other people to see the command.

	Revisions:

		v1 - Initial
		v1.01 - Fixed console says
]]
local function sayCmdCheck( userid, strText, bTeam )
	if userid == 0 then return strText end -- Ignore console

	local text = ULib.stripName( _PlayerInfo( userid, "name" ), strText )
	local argv = ULib.splitArgs( text )

	if ULib.sayCmds[ argv[ 1 ] ] then -- We've got a winner!
		local data = ULib.sayCmds[ argv[ 1 ] ]
		if data.access then
			if not data.ucl:query( userid, data.access ) then
				ULib.tsay( userid, "You do not have access to this command, " .. _PlayerInfo( userid, "name" ) .. "." )
				-- Print their name to intimidate them :)
				return "" -- Block from appearing
			end
		end

		local args = string.gsub( text, "^" .. argv[ 1 ] .. "%s*", "" ) -- Strip the caller command out
		local fn = data.fn
		local hide = data.hide
		table.remove( argv, 1 )
		local argc = table.getn( argv )

		fn( userid, args, argv, argc )
		if hide then return "" end
	end

	return strText
end
ULib.hookSay( sayCmdCheck, 2 )


--[[
	Function: addSayCommand
	
	Just like ULib's <CONCOMMAND()> except that the callback is called when the command is said in chat instead of typed in the console.

	Parameters:
		
		say_cmd - A command string for says. IE: "!kick", then when someone says "!kick", it'll call the callback.
		fn_call - The function to call when the command's called.
		cmd_help - *(Optional)* A help string for using the command.
		hide_say - *(Optional, defaults to false)* If true, will hide the chat message. Use this if you don't want other people to see the command.
		access - *(Optional, defaults to ACCESS_ALL)* The access required for using this command.
		ucl - *(Optional, defaults to ULib.mainUcl)* The ucl to query for access.

	Revisions:

		v1 - Initial
		v1.1 - Added access and ucl params.
]]
function ULib.addSayCommand( say_cmd, fn_call, cmd_help, hide_say, access, ucl )
	if not ULib.checkParams( { say_cmd, fn_call }, { "string", "function" } ) or ULib.trim( say_cmd ) == "" then return nil, ULib.ERR_ARGS end

	ucl = ucl or ULib.mainUcl
	ULib.sayCmds[ say_cmd ] = { fn=fn_call, hide=hide_say, cmd_help=cmd_help, access=access, ucl=ucl }
end

--[[
	Function: removeSayCommand
	
	Removes a say command.

	Parameters:
		
		say_cmd - The command string for says to remove.

	Revisions:

		v1.2 - Initial
]]
function ULib.removeSayCommand( say_cmd )
	if not ULib.checkParam( say_cmd, "string" ) then return nil, ULib.ERR_ARGS end

	ULib.sayCmds[ say_cmd ] = nil
end


--[[
	Function: stripName

	This strips a player name from a say string.

	Parameters:

		name - The player name
		str - The string to strip from

	Returns:

	The string minus the player name
]]
function ULib.stripName( name, str )
	if not ULib.checkParams( { name, str }, { "string", "string" } ) then return nil, ULib.ERR_ARGS end

	return string.sub( str, string.len( name ) + 3 ) -- + 3 to cover ": " and initial
end
