--[[
	Title: Utilities

	General GMod-related functions.
]]


--[[
	Table: timer

	This table is used for <addTimer()>.
]]--
ULib.timer = {}


--[[
	Function: addTimer

	Adds a timed function. Exactly like AddTimer() from GMod9, but *much* more robust and stable.
	It features error checking, and lets you know where the error is. (AddTimer from gmod doesn't tell you anything)

	Parameters:

		time - The interval on which the function will be called. IE, it will be called every x seconds from when you call this function.
		repeats - How many times to call this function, 0 = infinite.
		fn - The function to call
		args - Any number of extra arguments to pass to the callback function.
		
	Returns:

		Timer id on sucess, nil and error otherwise.
		
	Revisions:
	
		v1.2 - Initial
]]
function ULib.addTimer( time, repeats, fn, ... )
	local info = { time=time, last=_CurTime(), fn=fn, arg=arg, repeats=repeats }

	local key = 1
	while ULib.timer[ key ] ~= nil do
		key = key + 1
	end
	ULib.timer[ key ] = info
	return key
end


--[[
	Function: haltTimer

	Halts a timer added via ULib.addTimer

	Parameters:

		id - The timer id to halt

	Revisions:
	
		v1.2 - Initial
]]
function ULib.haltTimer( id )
	ULib.timer[ id ] = nil
end

local function updateTimers()
	local curtime = _CurTime()
	for key, info in pairs( ULib.timer ) do
		if info.last + info.time <= curtime then -- Needs to be called
			--ULib.print( "Running timerid " .. key )
			ULib.timercheck = true
			_RunString( "ULib.timer[ " .. key .. " ].fn( unpack( ULib.timer[ " .. key .. " ].arg ) ) ULib.timercheck = nil" ) -- This insulates us from errors.
			if ULib.timercheck then ULib.print( "[ULib timer] You probably just saw an error, this error came from timerid " .. key ) end
			ULib.timercheck = nil
			info.last = curtime -- Note, we bump it up even if it had an error this time.
			if info.repeats == 1 then ULib.haltTimer( key ) end
			if info.repeats ~= 0 then info.repeats = info.repeats - 1 end
		end
	end
end
AddThinkFunction( updateTimers )


--[[
	Function: execFile

	Executes a cfg file. The difference between this and "exec file" in console is that this function will run lua commands in the file, even on dedicated servers.
	It will treat "//" in lines as comments.

	Parameters:

		path - The location of the file to execute. Must be the full path from gmod9/ including the extension.

	Important:
	
		This function may not work for all lua commands, as it passes a userid of 0. ( IE: You couldn't use a command like "giveMeWeapon supahWep", because it wouldn't know who to give it to. )

	Revisions:

		v1 - Made it work, fixed lua commands.
		v1.2 - Fixed for empty file passes
]]
function ULib.execFile( file )
	if not _file.Exists( file ) then
		ULib.debug( "execFile was passed an invalid file!", ULib.ERR )
		return
	end

	local contents = ULib.stripComments( _file.Read( file ), "//" )
	if not contents or contents == "" then return end -- Empty file.

	local lines = ULib.explode( contents, "\n+" )

	for i=1, table.getn( lines ) do
		local location = string.find( lines[ i ], " " )
		if location then
			local command = ULib.trim( string.sub( lines[ i ], 1, location - 1 ) )
			if gConsoleCommands[ command ] then -- This is a lua command
				local args = ULib.trim( string.sub( lines[ i ], string.find( lines[ i ], " " ) + 1 ) )
   				gConsoleCommands[ command ]( 0, args ) -- Note that a userid of 0 may cause problems with some functions
   			else -- We're assuming it's just an average command to run
				_ServerCommand( lines[ i ] .. "\n" )
   			end
		else -- We're assuming it's just an average command to run
			_ServerCommand( lines[ i ] .. "\n" )
		end
	end
end


--[[
	Function: parseKeyValues

	Parses a keyvalue formatted string into a table.

	Parameters:

		str - The string to parse.

	Returns:

		The table, nil and error otherwise. *If you find you're missing information from the table, the file format might be incorrect.*
		
	Example format:
:test
:{
:	"howdy"   "bbq"
:
:	foo
:	{
:		"bar"   "false"
:	}
:
:}
]]
function ULib.parseKeyValues( str )
	if not ULib.checkParam( str, "string" ) then return nil, "Incorrect parameters!" end
	str = ULib.stripComments( str, "//" )
	
	local lines = ULib.explode( str, "\n" )
	lines = ULib.removeEmptyValues( lines ) -- Remove crap
	local pos = 1
	local t = {}

	while pos <= table.getn( lines ) do
		local args = ULib.splitArgs( lines[ pos ] )

		if table.getn( args ) == 1 then
			if pos ~= table.getn( lines ) and ULib.trim( lines[ pos + 1 ] ) == "{" then
				local endpos = pos + 2
				local inline = 0
				while true do -- We have breaks inside
					endpos = endpos + 1
					if endpos > table.getn( lines ) then return nil, "Bracket ( \"{\" and \"}\" ) mismatch!" end

					if ULib.trim( lines[ endpos ] ) == "{" then inline = inline + 1 end

					if ULib.trim( lines[ endpos ] ) == "}" then
						if inline > 0 then inline = inline - 1
						else break end
					end
				end

				t[ args[ 1 ] ] = ULib.parseKeyValues( table.concat( lines, "\n", pos + 2, endpos - 1 ) )

				pos = endpos
			else
				table.insert( t, args[ 1 ] )
			end
		elseif table.getn( args ) == 2 then
			t[ args[ 1 ] ] = args[ 2 ]
		else
			ULib.debug( "Incorrect syntax found while running parseKeyValues! Line was: \"" .. lines[ pos ] .. "\"", ULib.DEBUG )
		end
		pos = pos + 1
	end

	return t
end

--[[
	Function: makeKeyValues

	Makes a key values string from a table.

	Parameters:

		t - The table to make the keyvalues from. This can only contain tables, numbers, and strings.
		tab - *Only for internal use*, this helps make inline tables look better.
		completed - A list of table values that have already been parsed, this is *only for internal use* to make sure we don't hit an infinite loop.

	Returns:

		The string, nil and error otherwise.
		
	Notes:

		If you use numbers as keys in the table, just the values will be used.
		
	Example table format:
:{ test = { howdy = "bbq", foo = { bar = "false" } } }

	Example return format:
:test
:{
:	"howdy"   "bbq"
:
:	foo
:	{
:		"bar"   "false"
:	}
:
:}
]]
function ULib.makeKeyValues( t, tab, completed )
	if not ULib.checkParam( t, "table" ) then return nil, "Incorrect Parameters!" end

	tab = tab or ""
	completed = completed or {}
	if ULib.isInTable( completed, t ) then return "" end -- We've already done this table.
	table.insert( completed, t )
	str = ""

	for k, v in pairs( t ) do
		str = str .. tab
		if type( k ) ~= "number" then
			str = str .. ULib.serialize( k ) .. ULib.SPACE_METHOD
		end

		if type( v ) == "number" then v = tostring( v ) end
		if type( v ) == "table" then
			str = str .. ULib.NEWLINE .. tab .. "{" .. ULib.NEWLINE ..
				ULib.makeKeyValues( v, tab .. ULib.SPACE_METHOD, completed ) ..
				tab .. "}" .. ULib.NEWLINE

		else
			str = str .. ULib.serialize( v ) .. ULib.NEWLINE
		end
	end
	
	return str
end


--[[
	Function: play3DSound

	Plays a 3D sound, the further away from the point the player is, the softer the sound will be.

	Parameters:

		sound - The sound to play, relative to the sound folder.
		vector - The point to play the sound at.
		volume - *(Optional, defaults to 1)* The volume to make the sound.
		pitch - *(Optional, defaults to 1)* The pitch to make the sound, 1 = normal.
]]
function ULib.play3DSound( sound, vector, volume, pitch )
	volume = volume or 1
	pitch = pitch or 1

	local newent = ULib.makePoint( vector )
	_EntEmitSoundEx( newent, sound, volume, pitch )
end


--[[
	Variable: startTime
	
	This holds the global server uptime that the map started at.

	Revisions:

	v1 - Initial
]]
local startTime


--[[
	Function: refreshUptime
	
	This function is called every 5 seconds to update the global server uptime. *DO NOT CALL DIRECTLY*.

	Revisions:

	v1 - Initial
]]
local function refreshUptime()
	startTime = startTime or ULib.db.uptime or 0
	ULib.db.uptime = startTime + _CurTime()
end
ULib.addTimer( 5, 0, refreshUptime )


--[[
	Function: filesInDir
	
	Returns files in directory.

	Parameters:

		dir - The dir to look for files in.
		recurse - *(Optional, defaults to false)* If true, searches directories recursively.
		root - *INTERNAL USE ONLY* This helps with recursive functions.

	Revisions:

	v1 - Initial
]]
function ULib.filesInDir( dir, recurse, root )
	if not _file.Exists( dir ) or not _file.IsDir( dir ) then
		return nil, ULib.ERR_ARGS
	end

	local files = {}
	local relDir
	if root then
		relDir = string.gsub( dir, root .. "[\\/]", "" )
	end
	root = root or dir

	local result = _file.Find( dir .. "./*" )

	for i=1, table.getn( result ) do
		if result[ i ] ~= "." and result[ i ] ~= ".." then -- We just want to ignore these
			if _file.IsDir( dir .. "./" .. result[ i ] ) == true then
				if recurse then
					files = ULib.mergeTable( files, ULib.filesInDir( dir .. "/" .. result[ i ], recurse, root ) )
				end
			else
				if not relDir then
					table.insert( files, result[ i ] )
				else
					table.insert( files, relDir .. "/" .. result[ i ] )
				end
			end
		end
	end
	
	return files
end


--[[
	Function: makePoint

	Makes a point reference with an info_null. This is useful for things that can't receive a position and must have an entity.

	Parameters:

		pos - A vector3d position
		
	Returns:
	
		The entid of the point.
		
	Revisions:
	
		v1 - Initial
]]
function ULib.makePoint( pos )
	local newent = _EntCreate( "info_null" ) -- info_null will be "deleted" by the engine as soon as it's spawned, but still let's us access it's point.
	_EntSetPos( newent, pos )
	_EntSpawn( newent )
	
	return newent
end