----------------
--   Voting   --
----------------
local function voteCallback( id, name )
	if id then
		ULib.tsay( 0, "Option \"" .. tostring( name ) .. "\" won!" )
		ulx.logString( "Option \"" .. tostring( name ) .. "\" won the vote" )
	else
		ULib.tsay( 0, "No option won because no one voted!" )
		ulx.logString( "No one voted on previous poll." )
	end
end

function ulx.cc_vote( userid, args, argv, argc )
	if argc < 3 then
		ULib.tsay( userid, ulx.LOW_ARGS )
		return
	end

	local title = argv[ 1 ]
	table.remove( argv, 1 ) -- Get rid of the title

	local vote = ULib.VoteMenu:new( title, voteCallback, ulx.menu_skin )
	for _, v in ipairs( argv ) do
		vote:addOption( v )
	end
	vote:showVote( 0, 20 )
	ulx.logServAct( userid, "#A started a vote \"" .. args .. "\"" )
end

ulx.CONCOMMAND( "vote", ulx.cc_vote, "<question> <option 1> <option 2> ... - Takes up a vote with the specified options", ACCESS_VOTE, "!vote" )


local function voteMap2Callback( id, name )
	if not id then
		ULib.tsay( 0, "No option won because no one voted!" )
		ulx.logString( "No one voted on previous poll." )
		return
	end

	ULib.tsay( 0, "Option \"" .. tostring( name ) .. "\" won!" )
	if id == "no" then -- "No" won
		ulx.logString("Option \"" .. tostring( name ) .. "\" won" )
		return 
	end
	ulx.logString( "Votemap option \"" .. tostring( name ) .. "\" won. Changing map." )
	_ServerCommand( "changelevel " .. id .. "\n" )
end

function ulx.cc_voteMap2( userid, args, argv, argc )
	if argc < 1 then
		ULib.tsay( userid, ulx.LOW_ARGS )
		return
	end

	local vote
	if table.getn( argv ) == 1 then
		if not _file.Exists( "maps/" .. argv[ 1 ] .. ".bsp" ) then
			ULib.tsay( userid, "That map does not exist." )
			return
		end
		vote = ULib.VoteMenu:new( "Change to " .. argv[ 1 ] .. "?", voteMap2Callback, ulx.menu_skin )
		vote:addOption( "Yes", argv[ 1 ] )
		vote:addOption( "No", "no" )
	else
		vote = ULib.VoteMenu:new( "Change map to..", voteMapCallback, ulx.menu_skin )
		for _, v in ipairs( argv ) do
			if _file.Exists( "maps/" .. v.. ".bsp" ) then
				vote:addOption( v, v )
			end
		end
	end
	vote:showVote( 0, 20 )
	ulx.logServAct(userid, "#A started a map vote \"" .. ULib.implode(argv, ",") .. "\"")
end
ulx.CONCOMMAND( "votemap2", ulx.cc_voteMap2, "map [map2] .. - Gives the users a vote on the chosen maps (Yes/No question if only one specified)", ACCESS_VOTE, "!votemap2" )


local function voteKickCallback( id, name )
	if not id then
		ULib.tsay( 0, "No option won because no one voted!" )
		ulx.logString( "No one voted on previous poll." )
		return
	end

	ULib.tsay( 0, "Option \"" .. tostring( name ) .. "\" won!" )
	if id>_MaxPlayers() then -- "No" won
		ulx.logString("Votekick option \"" .. tostring( name) .. "\" won.")
		return 
	end 
	ulx.logString("Votekick option \"" .. tostring( name) .. "\" won. Kicking " .. ULib.playerString(id))
	
	ULib.kick( id )
end

function ulx.cc_voteKick( userid, args, argv, argc )
	if argc < 1 then
		ULib.tsay( userid, ulx.LOW_ARGS )
		return
	end
	
	local target, err = ULib.getUser( argv[ 1 ], _, ulx.ucl, userid )
	if not target then
		ULib.tsay( userid, err )
		return
	end

	local vote = ULib.VoteMenu:new( "Kick " .. _PlayerInfo( target, "name" ) .. "?", voteKickCallback, ulx.menu_skin )
	vote:addOption( "Yes", target )
	vote:addOption( "No", _MaxPlayers()+1 )
	vote:showVote( 0, 20 )
	ulx.logUserAct(userid, target, "#A started vote to kick #T")
end
ulx.CONCOMMAND( "votekick", ulx.cc_voteKick, "<user> - Gives the users a vote on whether or not to kick a player", ACCESS_VOTE, "!votekick" )


local function voteKickBanCallback( id, name )
	if not id then
		ULib.tsay( 0, "No option won because no one voted!" )
		ulx.logString( "No one voted on previous poll." )
		return
	end

	ULib.tsay( 0, "Option \"" .. tostring( name ) .. "\" won!" )
	if id>_MaxPlayers() then -- "No" won
		ulx.logString("Votekickban option \"" .. tostring( name ) .. "\" won.")
		return 
	end 
	ulx.logString("Votekickban option \"" .. tostring( name ) .. "\" won. Kicking and banning " .. ULib.playerString( id ))
	
	ULib.kickban( id )
end

function ulx.cc_voteKickBan( userid, args, argv, argc )
	if argc < 1 then
		ULib.tsay( userid, ulx.LOW_ARGS )
		return
	end

	local target, err = ULib.getUser( argv[ 1 ], _, ulx.ucl, userid )
	if not target then
		ULib.tsay( userid, err )
		return
	end

	local vote = ULib.VoteMenu:new( "Kickban " .. _PlayerInfo( target, "name" ) .. "?", voteKickBanCallback, ulx.menu_skin )
	vote:addOption( "Yes", target )
	vote:addOption( "No", _MaxPlayers()+1 )
	vote:showVote( 0, 20 )
	ulx.logUserAct(userid, target, "#A started a vote to kick and ban #T")
end
ulx.CONCOMMAND( "votekickban", ulx.cc_voteKickBan, "<user> - Gives the users a vote on whether or not to kickban a player", ACCESS_VOTE, "!votekickban" )


------------------
--Public votemap--
------------------
local maps = ulx.maps -- This is built in menus.lua
local ignoremaps = {}
local userMapvote = {} -- Indexed by userid.
local mapvotes = {} -- Indexed by map.
local timedVeto = nil

ulx.CONVAR( "votemap_enabled", "1", _, ACCESS_NONE ) -- Enable/Disable the entire votemap command
ulx.CONVAR( "votemap_mintime", "10", _, ACCESS_NONE ) -- Time after map change before votes count.
ulx.CONVAR( "votemap_waittime", "5", _, ACCESS_NONE ) -- Time before a user must wait before they can change their vote.
ulx.CONVAR( "votemap_successratio", "0.5", _, ACCESS_NONE ) -- Ratio of (votes for map)/(total players) needed to change map. (Rounds up)
ulx.CONVAR( "votemap_minvotes", "2", _, ACCESS_NONE ) -- Number of minimum votes needed to change map (Prevents llamas). This supercedes the above convar on small servers.
ulx.CONVAR( "votemap_vetotime", "30", _, ACCESS_NONE ) -- Time in seconds an admin has after a successful votemap to veto the vote. Set to 0 to disable.

function ulx.cc_votemapVeto( userid, args, argv, argc )
	if not timedVeto then
		ULib.tsay( userid, "There's nothing to veto!" )
		return
	end
	
	ULib.haltTimer( timedVeto )
	ULib.tsay( 0, "Votemap changelevel halted." )
	ulx.logServAct(userid, "#A vetoed the votemap")
end
ulx.CONCOMMAND( "veto", ulx.cc_votemapVeto, "This lets you halt a votemap changelevel.", ACCESS_VOTE, "!veto" )

function ulx.cc_votemapIgnore( userid, args, argv, argc )
	local map = argv[ 1 ]
	if string.sub( map, -4 ) == ".bsp" then
		map = string.sub( map, 1, -5 ) -- Take off the .bsp
	end

	mapid = ULib.findInTable( maps, map )
	if not mapid then
		return
	end

	ignoremaps[ mapid ] = true
end
ulx.CONCOMMAND( "votemap_ignore", ulx.cc_votemapIgnore, "", ACCESS_NONE )

function ulx.cc_votemap( userid, args, argv, argc )
	if not ULib.isStringTrue( ULib.getCvar( "ulx votemap_enabled" ) ) then
		ULib.tsay( userid, "The votemap command has been disabled by a server admin." )
		return
	end
	
	if timedVeto then
		ULib.tsay( userid, "You cannot vote right now, another map has already won and is pending approval." )
		return
	end

	if argc < 1 then
		ULib.console( userid, "Use \"votemap <id>\" to vote for a map. Map list:" )
		for id, map in ipairs( maps ) do
			if not ignoremaps[ id ] then
				ULib.console( userid, "  " .. id .. " -\t" .. map )
			end
		end
		return
	end
	
	local mintime = tonumber( ULib.getCvar( "ulx votemap_mintime" ) )
	mintime = mintime or 10 -- In case they tried assigning mintime a string...
	if _CurTime() < mintime * 60 then -- Minutes -> seconds
		ULib.tsay( userid, "Sorry, you must wait " .. mintime .. " minutes after a map change before you can vote for another map." )
		ULib.tsay( userid, "That means you must wait " .. ULib.getTimeStamp( mintime*60 - _CurTime(), true ) .. " more minutes." )
		return
	end

	if userMapvote[ userid ] then
		local waittime = tonumber( ULib.getCvar( "ulx votemap_waittime" ) )
		waittime = waittime or 5 -- In case they tried assigning it a string...
		if _CurTime() - userMapvote[ userid ].time < waittime * 60 then -- Minutes -> seconds
			ULib.tsay( userid, "Sorry, you must wait " .. waittime .. " minutes before changing your vote." )
			ULib.tsay( userid, "That means you must wait " .. ULib.getTimeStamp( waittime*60 - (_CurTime() - userMapvote[ userid ].time), true ) .. " more minutes." )
			return
		end
	end


	local mapid
	if tonumber( argv[ 1 ] ) then
		mapid = tonumber( argv[ 1 ] )
		if not maps[ mapid ] or ignoremaps[ mapid ] then
			ULib.tsay( userid, "Invalid map id!" )
			return
		end
	else
		local map = argv[ 1 ]
		if string.sub( map, -4 ) == ".bsp" then
			map = string.sub( map, 1, -5 ) -- Take off the .bsp
		end

		mapid = ULib.findInTable( maps, map )
		if not mapid then
			ULib.tsay( userid, "Invalid map!" )
			return
		end
	end
	
	if userMapvote[ userid ] then -- Take away from their previous vote
		mapvotes[ userMapvote[ userid ].mapid ] = mapvotes[ userMapvote[ userid ].mapid ] - 1
	end
	
	userMapvote[ userid ] = { mapid=mapid, time=_CurTime() }
	mapvotes[ mapid ] = mapvotes[ mapid ] or 0
	mapvotes[ mapid ] = mapvotes[ mapid ] + 1

	local minvotes = tonumber( ULib.getCvar( "ulx votemap_minvotes" ) )
	minvotes = minvotes or 0 -- In case they did something dumb.
	local successratio = tonumber( ULib.getCvar( "ulx votemap_successratio" ) )
	successratio = successratio or 0.5

	local votes_needed = math.ceil( math.max( minvotes, successratio * ULib.totalPlayers() ) ) -- Round up whatever the largest is.

	ULib.tsay( 0, _PlayerInfo( userid, "name" ) .. " voted for " .. maps[ mapid ] .. " (" .. mapvotes[ mapid ] .. "/" .. votes_needed .. "). Say \"!votemap " .. mapid .. "\" to vote for this map too." )

	if mapvotes[ mapid ] >= votes_needed then
		local vetotime = tonumber( ULib.getCvar( "ulx votemap_vetotime" ) )
		vetotime = vetotime or 30 -- In case they tried assigning it a string...
		
		local admins = {}
		for i=1, _MaxPlayers() do
			if _PlayerInfo( i, "connected" ) then
				if ulx.ucl.query( userid, ULib.GET_SUBCONCOMMAND( "ulx veto" ).access ) then
					table.insert( admins, i )
				end
			end
		end

		if table.getn( admins ) == 0 or vetotime < 1 then
			ULib.tsay( 0, "Vote for map " .. maps[ mapid ] .. " successful! Changing levels now." )
			ulx.logString("Votemap for " .. maps[ mapid ] .. " won.")
			_ServerCommand( "changelevel " .. maps[ mapid ] .. "\n" )
		else
			ULib.tsay( 0, "Vote for map " .. maps[ mapid ] .. " successful! Now pending admin approval. (" .. vetotime .. " seconds)" )
			for _, id in ipairs( admins ) do
				ULib.tsay( id, "To veto this vote, just say \"!veto\"" )
			end
			ulx.logString("Votemap for " .. maps[ mapid ] .. " won. Pending admin veto.")
			timedVeto = ULib.addTimer( vetotime, 1, _ServerCommand, "changelevel " .. maps[ mapid ] .. "\n" )
		end
	end
end
ulx.CONCOMMAND( "votemap", ulx.cc_votemap, "[<map name or id>] - Vote for a map. No args lists available maps.", ACCESS_ALL, "!votemap" )

function ulx.votemap_disconnect( name, userid ) -- We use this to clear out old people's votes
	if userMapvote[ userid ] then -- Take away from their previous vote
		mapvotes[ userMapvote[ userid ].mapid ] = mapvotes[ userMapvote[ userid ].mapid ] - 1
		userMapvote[ userid ] = nil
	end
end
HookEvent( "eventPlayerDisconnect", ulx.votemap_disconnect )
