--[[

	#################################################
	
		VMF CopyGun - Converts CopyGun Savefiles to VMFFiles
		
	#################################################
	
]]--

VMFCG.CGConverter = {};

-- ################### Creates a new object for a CG file to convert (object) @aVoN
function VMFCG.CGConverter:new(file)
	local obj = {}
	setmetatable(obj,self);
	self.__index = self;
	obj.content = VMFCG.explode(_file.Read(file),"\n");
	obj.entities = {}; -- Final entitys
	obj.set_targetname = {}; -- Entitys, which need a targetname
	obj.doors = {}; -- Saves an entitys targetname in here, to be able to rebind doors
	obj.buttons = {}; -- Stores the buttons for the doors above
	obj.welds = {}; -- Only saves the information, of a weld already exists (to remove duplicate welds)
	return obj;
end

-- ################### Parses an Entity of the cg save file (void) @aVoN
function VMFCG.CGConverter:ParseEntity(line,index)
	-- Remove { and < and } and > from each line
	local data = string.sub(line,2,-2);
	local keyvalues = {};
	local spawn_frozen = true;
	data = VMFCG.explode(data,"@");
	
	if(table.getn(data) == 4) then
		local key_names = {"classname","attach1","attach2"}
		local attach = {};
		for i=1,3 do -- The last (4th) table part is only crap :D
			local trimmed = string.sub(VMFCG.trim(data[i]),2); -- Remove the # infront of it
			local keyvalue = {key=key_names[i],value=trimmed};
			if(i > 1) then
				local id = tonumber(keyvalue.value);
				local targetname = "vmfcg_auto_"..id.."__respawn"; -- Autoname setting
				attach[i-1] = id;
				self.set_targetname[id] = targetname;
				keyvalue.value = targetname;
			end
			table.insert(keyvalues,keyvalue);
		end
		table.insert(keyvalues,{key="spawnflags",value="1"}); -- Don't collide
		-- Apply weld?
		if(self:WeldExists(attach[1],attach[2])) then return end;
		table.insert(self.welds,{attach[1],attach[2]});
	else
		-- Otherwise it's a random entity
		local key_names = {"classname","model","origin","angles"};
		for i=1,4 do -- The last (4th) table part is only crap :D
			local trimmed = string.sub(VMFCG.trim(data[i]),2); -- Remove the # infront of it
			if(i == 2) then
				trimmed = string.gsub(trimmed,"\\","/"); -- Change modelpath to /
			end
			table.insert(keyvalues,{key=key_names[i],value=trimmed});
		end
		local targetname = string.sub(VMFCG.trim(data[5]),2);
		if(targetname ~= "") then
			-- Possible Button
			if(string.find(targetname,"^button_")) then
				local shortened = string.sub(targetname,8);
				self.buttons[index] = shortened;
				table.insert(keyvalues,{key="spawnflags",value="256"});
			else
				-- Possible Door
				if(string.find(targetname,"^door_")) then
					local shortened = string.sub(targetname,6);
					self.doors[shortened] = index;
				end
			end
		end
		-- ###################  Now, handle special entities, like the Enhanced CopyGun does
		-- Prop Dynamic extensions
		if(keyvalues[1].value == "prop_dynamic" or keyvalues[1].value == "prop_dynamic_override") then
			keyvalues[1].value = "prop_dynamic_override";
			table.insert(keyvalues,{key="solid",value="6"});
			spawn_frozen = false;
		end
		-- For saved CombineBalls
		if(keyvalues[1].value == "prop_combine_ball") then
			keyvalues[1].value = "func_combine_ball_spawner"; -- Make them a combine ball spawner!
			table.insert(keyvalues,{key="ballcount",value="1"})
			table.insert(keyvalues,{key="minspeed",value="0"})
			table.insert(keyvalues,{key="maxspeed",value="0"})
			table.insert(keyvalues,{key="ballradius",value="90"})
			table.insert(keyvalues,{key="balltype",value="3"})
			table.insert(keyvalues,{key="allrespawntime",value="0.01"})
			spawn_frozen = false;
		end
		-- Rotateable doors
		if(keyvalues[1].value == "prop_door_rotating") then
			table.insert(keyvalues,{key="spawnflags",value="8192"})
			table.insert(keyvalues,{key="distance",value="90"})
			table.insert(keyvalues,{key="health",value="0"})
			table.insert(keyvalues,{key="returndelay",value="-1"})
			spawn_frozen = false;
		end
		-- Jeep
		if(keyvalues[1].value == "prop_vehicle_jeep") then
			local vehicle_script = VMFCG.defines.vehicles.scripts[keyvalues[2].value];
			if(not vehicle_script) then
				table.insert(keyvalues,{key="vehiclescript",value="scripts/vehicles/jeep_test.txt"})
			else
				table.insert(keyvalues,{key="vehiclescript",value=vehicle_script})
			end
			table.insert(keyvalues,{key="solid",value="6"})
			table.insert(keyvalues,{key="actionscale",value="1"})
			table.insert(keyvalues,{key="fadescale",value="1"})
			table.insert(keyvalues,{key="fademindist",value="-1"})
			spawn_frozen = false; -- never spawn a vehicle frozen or the game crashs!
		end
		-- The airboat
		if(keyvalues[1].value == "prop_vehicle_airboat") then
			table.insert(keyvalues,{key="solid",value="6"})
			table.insert(keyvalues,{key="vehiclescript",value="scripts/vehicles/airboat.txt"})
			table.insert(keyvalues,{key="actionscale",value="1"})
			table.insert(keyvalues,{key="fadescale",value="1"})
			table.insert(keyvalues,{key="fademindist",value="-1"})
		end
		-- The Pod
		if(keyvalues[1].value == "prop_vehicle_prisoner_pod") then
			table.insert(keyvalues,{key="solid",value="6"})
			table.insert(keyvalues,{key="vehiclescript",value="scripts/vehicles/prisoner_pod.txt"})
		end
		-- Magnetised stuff (otherwise it has infinite magnetise force) -reduce it to a very low level
		if(keyvalues[1].value == "phys_magnet") then
			table.insert(keyvalues,{key="forcelimit",value="100"})
			table.insert(keyvalues,{key="torquelimit",value="100"})
		end
		-- ###################  End of modifiing.
		if(spawn_frozen) then
			table.insert(keyvalues,{key="_spawnfrozen",value="true"}); -- Soawn any CG frozen
		end
	end
	table.insert(self.entities,{keyvalues=keyvalues,connections={}});
end

-- ################### Returns, if a weld already exists (boolean) @aVoN
function VMFCG.CGConverter:WeldExists(attach1,attach2)
	for _,v in self.welds do
		if((v[1] == attach1 and v[2] == attach2) or (v[1] == attach2 and v[2] == attach1)) then
			return true;
		end
	end
	return false;
end

-- ################### Returns a parsed CGsave file with all key/values (table) @aVoN
function VMFCG.CGConverter:Parse()
	local lines = table.getn(self.content);
	for i=1,(lines-1) do
		self:ParseEntity(self.content[i],i);
	end
	local height = tonumber(string.sub(self.content[lines],2,-2));
	-- Apply door/Buttons
	for k,v in self.buttons do
		if(self.doors[v]) then
			local door_id = self.doors[v];
			-- Apply connections  to the button, to make it posisble to open/close doors
			self.entities[k].connections = {};
			local targetname = "vmfcg_auto_"..door_id.."__respawn";
			table.insert(self.entities[k].connections,{key="addoutput",value="onplayeruse "..targetname..",setanimation,open"});
			table.insert(self.entities[k].connections,{key="addoutput",value="onplayeruse "..targetname..",setanimation,close,3"});
			-- Check, if the door entity has a targetname - Otherwise set it. Gets applyied below in the for-loop
			if(not self.set_targetname[door_id]) then
				self.set_targetname[door_id] = targetname;
			end
		end
	end
	-- When neccessary, set targetnames
	for k,v in self.set_targetname do
		table.insert(self.entities[k].keyvalues,{key="targetname",value=v});
	end
	local ent_objects = {};
	for k,v in self.entities do
		table.insert(ent_objects,VMFCG.Ent:CreatePseudo(k,v.keyvalues[1].value));
		for i,values in v.keyvalues do
			-- Classname just got added above
			if(i ~= 1) then
				ent_objects[table.getn(ent_objects)]:SetKeyValue(values.key,values.value);
			end
		end
		if(v.connections) then
			for _,connection in v.connections do
				ent_objects[table.getn(ent_objects)]:AddConnection(connection.key,connection.value);
			end
		end
	end
	return ent_objects,height;
end

-- ################### Converts the final CGS to a VMF with the VMF Save Engine
function VMFCG.CGConverter.ConvertFiles()
	local search_folder = "cg_saves/";
	if(_file.Exists("lua/copygun_config.lua")) then
		_OpenScript("copygun_config.lua");
		search_folder = DEFAULT_SAVE_PARTIALDIR;
	end
	if(_file.Exists(string.gsub(search_folder,"[/\\]$",""))) then
		local files = _file.Find(search_folder.."*.cgs")
		-- ################### This part fully converts the CGS files to VMF's
		for _,file in files do
			local new_filename = string.sub(file,1,-5);
			if(not _file.Exists(VMFCG.config.save_folder.."/patterns/"..new_filename..".vmf")) then
				local CGS = VMFCG.CGConverter:new(search_folder..""..file);
				local saver = VMFCG.saver:new(CGS:Parse());
				saver:GenerateVMF(new_filename,nil);
			end
		end
		-- ################### Now, if it's the Enhanced CopyGun, add the databases of them to the VMFCopyGun ones.
		local ECGDB = {"copygun_patterns","copygun_users","copygun_delete","copygun_public"};
		local VMFCGDB = {"patterns","users","delete","public"};
		for i,name in ECGDB do
			if(_file.Exists(name..".db")) then
				local db = dbloader:New(name..".db");
				db:Load();
				local content = db:GetTable();
				for k,v in content do
					-- Only add entry, when it doesn't exists in our db :D
					if(not VMFCG.database[VMFCGDB[i]]:Get(k)) then
						VMFCG.database[VMFCGDB[i]]:Set(k,v);
					end
				end
				VMFCG.database[VMFCGDB[i]]:Save();
			end
		end
		VMFCG.menu.ReloadMenu(); -- Update menu
	end
end
