--[[

	#################################################
	
		VMF CopyGun - SaveEngine
		
	#################################################
	
]]--

VMFCG.saver = {};

-- #############################################
-- ############### Saver Object ######################
-- #############################################


-- ################### Constructs the saver object with VMFCopyGun-style saveed entities (ent Class) (object) @aVoN
function VMFCG.saver:new(entities,height)
	local obj = {}
	setmetatable(obj,self);
	self.__index = self;
	-- Needed, or modifications on this obj.entities table affect the original entites table too (seletec props)
	local n_e = {};
	for _,v in entities do
		table.insert(n_e,v);
	end
	obj.targetnames = {};
	obj.targetname_replace = {};
	obj.entities = n_e;
	obj.position_sub = vector3(0,0,0); -- Saves, the vector3 of an entity's position (the ent, which got selected first by the user) and calculates the difference
	obj.height_add = height or 0; -- Same like the position, but adds a Z value for height.
	obj.calculated_translation = false; -- Handles wether the values above get recalculated or not
	obj:sortEntities(); -- Get all entities and sort them
	obj:RetrieveTargetNames(); -- Creates targetnames, so the stuff gets attached, or loading will fail
	return obj;
end

-- ################### Counts the ammount of stored entitys in this object (integer) @aVoN
function VMFCG.saver:EntityCount()
	return table.getn(self.entities);
end

-- ################### Same as above, but with Entity Id's instead of real Entity-Informations (void) @aVoN
function VMFCG.saver:newById(entities)
	local return_entities = {};
	for _,v in entities do
		if(VMFCG.Entities.Exists(v)) then
			table.insert(return_entities,VMFCG.Entities[v]);
		else
			table.insert(return_entities,VMFCG.Ent:GetById(v));
		end
	end
	return VMFCG.saver:new(return_entities);
end


-- #############################################
-- ############### Target Naming #####################
-- #############################################

-- ################### Gets the targetname of an entity (string) @aVoN
function VMFCG.saver:GetTargetName(id)
	id = VMFCG.Entities.getEntId(id);
	if(self.targetnames[id]) then
		return self.targetnames[id];
	else
		return; -- Otherwise, set the targetname to 0 for VMFCG kill engine, or it gets autoremoved :( 
	end
end

-- ################### Sets correct targetnames for entitys (void) @aVoN
function VMFCG.saver:RetrieveTargetNames()
	local entities = self.entities;
	for _,name in VMFCG.defines.fixup.targetnames do
		for _,v in entities do
			if(v:HasKey(name)) then
				local name = v:GetKeyValue(name);
				if(name ~= "" and not self.targetname_replace[name]) then
					local ids = {VMFCG.Entities.getEntId(name)};
					if(ids[1] ~= 0) then
						-- To minimize the occurence of one certain problem, i just can't find out, why it happens, i added this math.random part, so copieing will ALWAYS return a random number on multiple targetnames which are the same
						local key = table.getn(ids);
						if(key > 1) then
							key = math.random(1,key);
						end
						local new_name = "vmfcg_auto_"..ids[key].."__respawn";
						self.targetname_replace[name] = new_name;
						for _,id in ids do
							self.targetnames[id] = new_name;
						end
					end
				end
			end
		end
	end
end


-- #############################################
-- ############### Position recalculations ################
-- #############################################


-- ################### Calculates the needed height and position for a VMF which shall be saved (void) @aVoN
function VMFCG.saver:PositionHeight(ent)
	-- Trace to the ground
	if(not self.calculated_translation) then
		self.calculated_translation = true;
		self.position_sub = ent:GetPos();
		if(ent:Exists()) then
			local t = VMFCG.TraceLine:New(self.position_sub,vector3(0,0,-1),VMFCG.config.trace_default_length,ent:GetId());
			t:CollisionGroup(COLLISION_GROUP_DEBRIS); -- Only let it hit the world
			if(t:Fraction() == 1) then
				-- For some special entitys, which's origin are stuck in ground.. Like most of phoenix/gmod and vmod :(
				-- Go first 100 units above, start a traceline there and recalculate a new endpos
				t = VMFCG.TraceLine:New(vecAdd(self.position_sub,vector3(0,0,100)),vector3(0,0,-1),100,ent:GetId());
				t:CollisionGroup(COLLISION_GROUP_DEBRIS);
				self.height_add = -(100 - 100*t:Fraction());
			else
				self.height_add = VMFCG.config.trace_default_length*t:Fraction();
			end
		end
	end
end


-- #############################################
-- ############### Entity sorting for correct spawnorder ########
-- #############################################


-- ##################### Get all entities needed entities and sorts them for saving @aVoN
function VMFCG.saver:sortEntities()
	local selected = self.entities;
	local phys = {};
	local props = {};
	local ropes = {};
	local parented = {};
	
	-- Getting them sorted
	for _,v in selected do
		local done = false;
		local cln = v:GetClass();
		if(string.find(cln,"phys_") and cln ~= "phys_magnet") then
			table.insert(phys,v);
			done = true;
		end
		if(not done) then
			if(cln == "keyframe_rope") then
				table.insert(ropes,v);
				done = true;
			end
		end
		if(not done) then
			local parent = v:GetParent();
			if(parent ~= nil and parent ~= 0) then
				table.insert(parented,v);
				done = true;
			end
		end
		if(not done) then
			table.insert(props,v);
		end
	end
	-- And create a new sorted entity list
	local final_entities = {};
	final_entities = VMFCG.merge(final_entities,props);
	final_entities = VMFCG.merge(final_entities,parented);
	final_entities = VMFCG.merge(final_entities,phys);
	final_entities = VMFCG.merge(final_entities,ropes);
	self.entities = final_entities;
end


-- #############################################
-- ############### VMF Creation parts ##################
-- #############################################


-- ##################### Make a string from a key-value pair (string) @H-=NooB=-
function VMFCG.saver:makeKeyValueStr(k,v)
	local str = "";
	local key = tostring(k);
	local value = tostring(v);
	if (key ~= "" and value ~= "") then
		str = '	"' .. key .. '" "' .. value .. '"\n'; 
	end
	return str;
end

-- ################### Get the VMF's required header (string) @H-=NooB=-
function VMFCG.saver:getVMFHeader()
	local header = "";
	if _file.Exists("lua/VMFCopyGun/data/vmf_header_dump.dat") then
		header = _file.Read("lua/VMFCopyGun/data/vmf_header_dump.dat");
		VMFCG.debug("VMFCG.saver:getVMFHeader","Header loaded successfuly!");
	else
		VMFCG.debug("VMFCG.saver:getVMFHeader","Missing file 'vmf_header_dump.dat'!");
	end
	return header;
end

-- ################### Get the VMF's required footer (string) @aVoN
function VMFCG.saver:getVMFFooter()
	local footer = "";
	if (_file.Exists("lua/VMFCopyGun/data/vmf_footer_dump.dat")) then
		footer = _file.Read("lua/VMFCopyGun/data/vmf_footer_dump.dat");
		VMFCG.debug("VMFCG.saver:getVMFFooter","Footer loaded successfuly!");
	else
		VMFCG.debug("VMFCG.saver:getVMFFooter","Missing file 'vmf_footer_dump.dat'!");
	end
	return footer;
end

-- ################### Get the VMF's metainfo footer (string) @aVoN
function VMFCG.saver:getVMFMeta()
	local meta = "";
	if (_file.Exists("lua/VMFCopyGun/data/vmf_metadata_dump.dat")) then
		meta = _file.Read("lua/VMFCopyGun/data/vmf_metadata_dump.dat");
		VMFCG.debug("VMFCG.saver:getVMFMeta","Metainfos loaded successfuly!");
	else
		VMFCG.debug("VMFCG.saver:getVMFMeta","Missing file 'vmf_metadata_dump.dat'!");
	end
	return meta;
end

-- ########################### First go threw all key/values and search for stuff like _parent etc to align right values for saving and  Look for some special keys (table) @ aVoN
function VMFCG.saver:parseKeyValues(entity)
	local id = entity:GetId();
	local keyvalues = entity:GetAllKeyValues();
	keyvalues["id"] = nil;
	if(entity:Exists()) then
		-- Change targetname to the entityID they currently have to prevent mutlisave errors.
		for _,v in VMFCG.defines.fixup.targetnames do
			if(keyvalues[v]) then
				keyvalues[v] = self:GetTargetName(keyvalues[v]);
			end
		end
		-- ###########################  Set targetnames
		keyvalues["targetname"] = self:GetTargetName(id);
	end
	-- ########################### Calculate the correct position for some phys_ entitys
	if(keyvalues["classname"] ~= "vmf_global") then -- Don't correct the originin the the vmf_global entity
		for _,v in VMFCG.defines.fixup.vectors do
			if(keyvalues[v]) then
				local new_pos = VMFCG.explode(keyvalues[v]," ");
				local z_difference = new_pos[3]-tonumber(self.position_sub.z);
				new_pos[1] = new_pos[1]-tonumber(self.position_sub.x);
				new_pos[2] = new_pos[2]-tonumber(self.position_sub.y);
				new_pos[3] = new_pos[3]-tonumber(self.position_sub.z)+tonumber(self.height_add);
				if(keyvalues["classname"] == "weapon_swep") then
					if(math.floor(z_difference) <= 1) then
						new_pos[3] = new_pos[3] + 10; -- Or it will spawn in ground = non-visible
					end
				end
				keyvalues[v] = vecString(vector3(new_pos[1],new_pos[2],new_pos[3]));
			end
		end
	end
	-- ########################### The classname at the very beginning please!...
	if(keyvalues["classname"]) then
		-- Save prop_dynamics always as prop_dynamic_override!
		if(keyvalues["classname"] == "prop_dynamic") then
			keyvalues["classname"] = "prop_dynamic_override";
		end
	end
	return keyvalues;
end

-- ########################### Goes threw all connections and changes targetnames in it to the correct one (table) @ aVoN
function VMFCG.saver:parseConnections(entity)
	local connections = entity:GetConnections();
	if(table.getn(connections) > 0) then
		-- I really hate wildcards. But thanks to Jintos battery_turret system. I never knew, outputs can have targetname wildcards :D
		local wild_carded = {};
		for k,v in connections do
			if(string.find(v.value,"%*")) then
				table.insert(wild_carded,connections[k]);
				connections[k] = nil;
			end
		end
		for k,v in self.targetname_replace do
			for ck,cv in connections do
				connections[ck].value = string.gsub(cv.value,k,v);
			end
		end
		if(table.getn(wild_carded) > 0) then
			local additional_connections = {};
			for _,v in wild_carded do
				-- Fetch targetname and replace the connection wildcard "*" by a lua string."function" wildcard => "."
				local targetname_end = string.find(v.value,",");
				local pattern = VMFCG.trim(string.gsub(string.sub(v.value,1,targetname_end-1),"%*","."));
				-- Find matching targetname
				for k,name in self.targetname_replace do
					if(string.find(k,pattern)) then
						local new_connection = name..string.sub(v.value,targetname_end);
						table.insert(additional_connections,{key=v.key,value=new_connection});
					end
				end
			end
			connections = VMFCG.merge(connections,additional_connections);
		end
	end
	return connections;
end

-- ########################### Make a VMF Capable string from an entities key-value pairs (string) @H-=NooB=- fully rewritten by aVoN
function VMFCG.saver:makeEntity(entity,index)
	local keyvalues = self:parseKeyValues(entity);
	local connections = self:parseConnections(entity);
	local ent = 'entity\n{\n';
	ent = ent .. '	"id" "' .. index .. '"\n'
	-- Correct this attach-entity, when it is a bugbait!
	if(keyvalues["model"] == "models/weapons/w_bugbait.mdl" and keyvalues["classname"] == "prop_physics" and entity:GetParent() ~= 0) then
		keyvalues["classname"] = "prop_dynamic_override";
		keyvalues["solid"] = 0;
		keyvalues["rendermode"] = 3;
		keyvalues["renderamt"] = 0;
		-- And update Entity Cache
		if(entity:Exists()) then
			local id = entity:GetId();
			VMFCG.Entities[id]:SetClass("prop_dynamic_override")
			VMFCG.Entities[id]:SetKeyValue("solid",0);
			VMFCG.Entities[id]:SetKeyValue("rendermode",3);
			VMFCG.Entities[id]:SetKeyValue("renderamt",0);
			VMFCG.Entities[id]:SetKeyValue("renderamt",0);
			VMFCG.Entities[id]:SetParent(entity:GetParent());
		end
	end
	-- ########################### First KV followed by targentname,origin and angle! Needed, because angles need to get set before it gets parented or we cant apply the angles anymore the irght way
	local first_kv = {"classname","targetname","origin","angles","hingeaxis","slideaxis","attach1","attach2","parentname"};
	for _,v in first_kv do
		if(keyvalues[v]) then
			ent = ent .. self:makeKeyValueStr(v,keyvalues[v]);
			keyvalues[v] = nil;
		end
	end
	-- ########################### Create Final K/v string now
	for k, v in keyvalues do
		ent = ent .. self:makeKeyValueStr(k,v);
	end
	-- ########################### Add connections
	if(table.getn(connections) > 0) then
		ent = ent..'	connections\n	{\n';
		for k,v in connections do
			ent = ent .. '	'..self:makeKeyValueStr(v.key,v.value);
		end
		ent = ent..'	}\n';
	end
	ent = ent .. '	editor\n	{\n		"color" "220 30 220"\n		"visgroupshown" "1"\n		"visgroupautoshown" "1"\n	}\n}\n';
	return ent;
end

-- ######################### Generate a VMF file from an entity list (string) @H-=NooB=- (rewritten and changed by aVoN)
function VMFCG.saver:GenerateVMF(file,userid)
	local entities = self.entities;
	if (table.getn(entities) == 0) then
		VMFCG.debug("VMFCG.saver:GenerateVMF","No entities given");
		return false;
	end
	local VMF = self:getVMFHeader().."\n";
	if (not VMF) then return false; end; -- Error already in self:getVMFHeader() :D -- Cant create a valid VMF without the correct header
	local use_selection = false;
	if(userid) then
		if(VMFCG.count(VMFCG.selection.users[userid].entities) > 0) then
			use_selection = true;
		end
	end
	if(use_selection) then
		-- Find the first entity, a user has selected and create for this the correct pos_sub and height_add
		for _,v in VMFCG.selection.users[userid].entities do
			self:PositionHeight(VMFCG.Ent:GetById(v))
			break;
		end
	else
		-- ... otherwise take the first entity in the entity list
		self:PositionHeight(entities[1]);
	end
	local meta_content = {};
	-- The entities meta is just a simple counter. The abs_pos is for absolute spawning, and map identifies this VMF as an absolutespawnable contraption on this map
	meta_content["origin"] = vecString(vecSub(self.position_sub,vector3(0,0,self.height_add)));
	meta_content["map"] = _GetCurrentMap();
	meta_content["entities"] = self:EntityCount();
	if(VMFCG.config.add_meta) then
		meta_content["server"] = _GetConVar_String("hostname");
		meta_content["time"] = _CurTime();
		meta_content["filename"] = file..".vmf";
		meta_content["description"] = file..".vmf"; -- Same as file, but for VMFLoader
		if(_PlayerInfo(userid,"connected")) then
			meta_content["author"] = _PlayerInfo(userid,"name");
			meta_content["author_contact"] = _GetClientConVar_String(userid,"stmlgn").." (SteamAccount)";
			meta_content["steamid"] = _PlayerInfo(userid,"networkid");
		end
	end
	local meta_entity = VMFCG.Ent:CreatePseudo(0,"vmf_global");
	for k,v in meta_content do
		meta_entity:SetKeyValue(k,v);
	end
	table.insert(entities,meta_entity);
	-- ######################### Entity creation part
	for k, v in entities do
		VMF = VMF .. self:makeEntity(v,k+2);
	end
	-- ######################### Metadatas
	VMF = VMF .. self:getVMFFooter();
	_file.Write(VMFCG.config.save_folder.."/patterns/"..file..".vmf",VMF);
end


-- #############################################
-- ############### Parse entitys to a VMFLoader capable table #####
-- #############################################


-- ######################### Generates a table of all neccessary key/values for the VMFLoader (string) @aVoN
function VMFCG.saver:VMFLoaderPreviewMakeEntity(entity)
	local id = entity:GetId();
	local keyvalues = self:parseKeyValues(entity);
	local VMFLoader_style_kv = {};
	local first_kv = {"classname","targetname","origin","angles","hingeaxis","slideaxis","attach1","attach2","parentname"};
	for _,v in first_kv do
		if(keyvalues[v]) then
			table.insert(VMFLoader_style_kv,{key = v,value = keyvalues[v]});
			keyvalues[v] = nil;
		end
	end
	for k,v in keyvalues do
		table.insert(VMFLoader_style_kv,{key = k,value = v});
	end
	connections = self:parseConnections(entity);
	return {connections = connections,keyvalues = VMFLoader_style_kv};
end

-- ######################### Generates all props for the VMFLoader preview (string) @aVoN
function VMFCG.saver:VMFLoaderPreviewEntities(userid)
	local entities = self.entities;
	local ent_list = {};
	if(userid) then
		-- Find the first entity, a user has selected and create for this the correct pos_sub and height_add
		for _,v in VMFCG.selection.users[userid].entities do
			self:PositionHeight(VMFCG.Ent:GetById(v));
			break;
		end
	else
		-- ... otherwise take the first entity in the entity list
		self:PositionHeight(entities[1]);
	end
	for _, v in entities do
		table.insert(ent_list,self:VMFLoaderPreviewMakeEntity(v));
	end
	return ent_list;
end


-- #############################################
-- ############### Namefilter for targetnames #############
-- #############################################

-- ######################### For name filter. Checks, if a given key is a key with a targetname in it!! @aVoN
function VMFCG.saver.validTargetnameKey( key )
	for _,v in VMFCG.defines.fixup.targetnames do
		if(string.lower(v) == string.lower(key)) then
			return true;
		end
	end
	return false;
end

-- ######################### Like the function in VMFLoader but much faster!!  @aVoN
function VMFCG.saver:NameFilter(entities)
	if(VMFCG.core.validVMFLoader()) then
		 -- Find entity names.
	    local targetnames = {};
	    for _, entity in entities do
	        local targetname = VMFLoader:FindKV( entity.keyvalues, "targetname" );
	        if(targetname) then
				table.insert(targetnames, VMFCG.trim(targetname.value));
	        end
	    end
	    targetnames = VMFCG.unique(targetnames); -- Make unique table
	    -- Go threw all key/values and connection and check for targetnames
		for _, entity in entities do
			-- Replace keyvalues.
			for _, kv in entity.keyvalues do
				if(VMFCG.saver.validTargetnameKey( kv.key )) then
					for _,name in targetnames do
						if(kv.value == name) then
							kv.value = VMFLoader.SpawnCounter.."__"..name;
							break;
						end
					end
				end
			end
			-- Replace connections.
			for _, kv in entity.connections do
				-- I hate fucking wildcards
				local replace_wildcard = string.find(kv.value,"*") or 0;
				for _,name in targetnames do
					local replace_name = name;
					local search_pattern = name;
					if(replace_wildcard ~= 0) then
						if(string.len(name) >= replace_wildcard) then
							local part1 = string.sub(name,1,replace_wildcard-1);
							local part2 = string.sub(name,replace_wildcard+1);
							replace_name = part1.."*"..part2;
							search_pattern = part1.."%*"..part2;
						end
					end
					if(string.find(kv.value,"^"..search_pattern..",")) then
						kv.value = string.gsub(kv.value,search_pattern,VMFLoader.SpawnCounter.."__"..replace_name);
						break;
					end
				end
			end
		end
	end
end
