--[[

	#############################################
	
		VMF CopyGun - Core functions
		
	#############################################
	
]]--

VMFCG.core = VMFCG.core or {}
VMFCG.core.override = VMFCG.core.override or {};
VMFCG.core.last_no_vmfloader_warning = 0; -- Handles, when the script warned the user, he isn't using an actual version of VMFLoader

-- ##############  Constructor for Override-Functions @ aVoN
-- Metatable for the override function subsystem. When the VMFCG.code table gets e.g. indexed with an overrid-function like VMFCG.code._EntSetKeyValue, the script checks the table VMFCG.core.override instead for it and returns this function.
setmetatable(
	VMFCG.core,
	{__index = 
		function (t,n)
			if(not rawget(t,n) and VMFCG.core.override[n]) then
				return VMFCG.core.override[n];
			end
			return nil;
		end
	}
);

-- ############## Fetches the materials (void) @aVoN
function VMFCG.core.getMaterialNames()
	-- Fetch all on the server installed materials
	local materials = _file.Read("settings/gmod_materials.txt")
	materials = string.sub(materials,string.find(materials,"{")+1,string.find(materials,"}")-1);
	materials = string.gsub(materials,"\r",""); -- Remove windows return-spaces
	material_table = {};
	for _,v in VMFCG.explode(materials,"\n") do
		local found_commend = string.find(v,"//");
		if(found_commend ~= nil) then
			v = string.sub(v,1,string.find(v,"//")-found_commend); -- Remove comments
		end
		if(string.gsub(v,"\"[%s]+\"","") ~= "") then
			v = string.gsub(v,"\"[%s]+\"","___"); -- Remove spaces
			v = string.gsub(v,"\"",""); -- Now the last resisting double-quotes \"
			table.insert(VMFCG.defines.materials,VMFCG.explode(v,"___")[2]);
		end
	end
	VMFCG.debug("VMFCG.core.getMaterialNames","Fetched Material Table From gmod_materials.txt");
end

-- ############## Warns a user, when he uses no or an old version of VMFLoader(void) @aVoN
function VMFCG.core.validVMFLoader()
	local hasVMF = true;
	if(VMFLoader) then
		if(VMFLoader.version) then
			if(not (VMFLoader.version >= 0.13)) then
				hasVMF = false;
			end
		else
			hasVMF = false;
		end
	else
		hasVMF = false;
	end
	if(not hasVMF) then
		if(VMFCG.config.show_no_vmfloader_warning) then
			VMFCG.Hud.Message(0,"The VMFCopyGun needs atleast VMFLoader v0.13. Downloadlink in chatarea!");
			if((VMFCG.core.last_no_vmfloader_warning + 15) <= _CurTime()) then
				_PrintMessageAll(HUD_PRINTTALK,"VMFLoader: http://forums.facepunchstudios.com/showthread.php?t=72955");
				VMFCG.core.last_no_vmfloader_warning = _CurTime();
			end
		end
	end
	return hasVMF;
end

-- #############################################
-- ############### Override Functions part ################
-- #############################################


-- ############## Overrides one function with another and executes both, when the original function name gets called (void) @aVoN
function VMFCG.core.overrideFunctionAdd(original_function,new_function)
	if(VMFCG.core.override[original_function]) then
		-- First remove this function, if exists before
		VMFCG.core.overrideFunctionRemove(original_function);
	end
	_RunString("VMFCG.core.override['"..original_function.."'] = "..original_function);
	_RunString(original_function.." = function(...) if("..new_function.."(unpack(arg)) ~= false) then VMFCG.core.override['"..original_function.."'](unpack(arg)) end end ");
	VMFCG.debug("VMFCG.core.overrideFunctionAdd","Adding override-function "..new_function.." for "..original_function);
end

-- ############## Removes an function override (void) @aVoN
function VMFCG.core.overrideFunctionRemove(function_name)
	if(VMFCG.core.override[function_name]) then
		_RunString(function_name.." = VMFCG.core.override['"..function_name.."']");
		VMFCG.core.override[function_name] = nil;
		VMFCG.debug("VMFCG.core.overrideFunctionRemove","Removing override-function "..function_name);
	end
end

-- ############## Removes all function overrides (void) @aVoN
function VMFCG.core.overrideFunctionRemoveAll()
	VMFCG.debug("VMFCG.core.overrideFunctionRemoveAll","Removing all override-functions");
	for k,_ in VMFCG.core.override do
		VMFCG.core.overrideFunctionRemove(k);
	end
end

-- ############## Overwrites the original _EntSetKeyValue() function with this one, so the script automaticly detects changes of key/values by third_party scripts (void) @aVoN
function VMFCG.core._EntSetKeyValue_new(id,key,value)
	if(not _EntExists(id)) then return nil end;
	if(k ~= "origin" and k ~= "angles") then
		local k = string.lower(key);
		local v = value;
		local class = _EntGetType(id);
		if(class == "weapon_swep" and k == "script") then
			VMFCG.AddTimerOnce("VMFCG.SWEPHook._PlayerGiveSWEP_timer",0.1,1,VMFCG.SWEPHook._PlayerGiveSWEP_timer,0,value,id);
		end
		-- Make logic_auto entitys NOT disappear after creation (disable setting of the spawnflag for it)
		if(class == "logic_auto" and k == "spawnflags") then
			return false;
		end
		-- For those, who use the windows filesystem with \ instead of /, which is really annoying!
		if(key == "model") then
			v = string.gsub(v,"\\","/");
		end
		-- Autoparenting of the physentity to it's attach2 entity.
		if(string.find(key,"attach[12]")) then
			-- Pulley, because of his force-point and phys_thruster because he can't apply force then anymore :(
			if(class ~= "phys_pulleyconstraint" and class ~= "phys_thruster" and _EntGetParent(id) == 0) then
				VMFCG.AddTimerOnce("VMFCG.core.AttachEntityAutoParenting:"..id,0.03,1,VMFCG.core.AttachEntityAutoParenting,id,value);
			end
		end
		if(k == "targetname") then
			-- And fixup targetnames ;)
			VMFCG.Entities.ChangeTargetName(id,value);
		end
		-- Don't save stuff like origin, because it can get fetched by _EntGetPos/_EntGetAngAngles
		if(value == "") then
			v = nil;
		end
		-- Apply thse special key/values directly threw the ent-class
		if(key == "_gravity" or key == "_mass" or key == "_movetype" or key == "_material" or key == "_collision" or "materialoverride") then
			VMFCG.Entities[id]:SetKeyValue(key,value,true);
			return false;
		else
			VMFCG.Entities[id]:SetKeyValue(key,v);
		end
	else
		if(VMFCG.core.isRoped(id) and VMFCG.config.anti_rope_fuckup) then return false end; -- Checks an entity for beeing roped and disallows setting of the angle or position to avoid the RopeFuckUp bug
		if(k == "angles") then
			if(VMFCG.Entities.Exists(id)) then
				if(VMFCG.Entities[id]:HasKey("hingeaxis") or VMFCG.Entities[id]:HasKey("slideaxis")) then
					-- Disallow the setting of an angle when hingeaxis and slide axis still exist as a key value!
					return false;
				end
			end
		end
	end
end

-- ############## Parents an entity automaticly to it's attach2 entity, when it is a phys entity. Gets called by VMFCG.core._EntSetKeyValue_new
function VMFCG.core.AttachEntityAutoParenting(entity,attach)
	if(_EntGetParent(entity) == 0) then
		local attach_id = VMFCG.Entities.getEntId(attach);
		if(attach_id ~= 0) then
			_EntSetParent(entity,attach_id);
			VMFCG.debug("VMFCG.core.AttachEntityAutoParenting","Setting entity "..attach_id.." as parent for '".._EntGetType(entity).."' "..entity);
		end
	end
end

-- ############## Checks, if an entity is roped (void) @aVoN
function VMFCG.core.isRoped(e)
	for k,v in VMFCG.Entities do
		if(type(k) == "number") then
			local class = v:GetClass();
			if(class == "phys_slideconstraint" or class == "phys_lengthconstraint" or class == "phys_spring") then
				if(v:HasKey("attach1") or v:HasKey("attach2")) then
					local attach1 = VMFCG.Entities.getEntId(v:GetKeyValue("attach1"));
					local attach2 = VMFCG.Entities.getEntId(v:GetKeyValue("attach2"));
					if(e == attach1 or e == attach2) then
						return true;
					end
				end
			end
		end
	end
	return false;
end

-- ############## Autoprecaches a model if neccessary to avoid servercrashes by thirdparty scripts (void) @ aVoN
function VMFCG.core._EntSetModel_new(id,model)
	if(not VMFCG.Ent.Precached[model]) then
		_EntPrecacheModel(model);
		VMFCG.Ent.Precached[model] = true;
	end
end

-- ############## Same for _EntSetMaterial (void)  @ aVoN
-- And declare the new function for it
function VMFCG.core._EntSetMaterial_new(id,mat)
	if(not _EntExists(id)) then return nil end;
	local m = mat;
	if(mat == "") then
		m = nil;
	end
	VMFCG.Entities[id]:SetKeyValue("_material",m);
	VMFCG.Entities[id]:SetKeyValue("materialoverride",m);
end

-- ############## Fetch _EntSetAng and _EntSetAngAngle values handles, if it get's set or not (on phys_slideconstraint and phys_hinge) (void) @ aVoN
function VMFCG.core._EntSetAng_and_EntSetAngAngle_new(id,ang)
	if(not _EntExists(id)) then return nil end;
	if(VMFCG.core.isRoped(id) and VMFCG.config.anti_rope_fuckup) then return false end; -- Checks an entity for beeing roped and disallows setting of the angle to avoid the RopeFuckUp bug
	if(VMFCG.Entities.Exists(id)) then
		if(not (VMFCG.Entities[id]:GetKeyValue("hingeaxis") == "" and VMFCG.Entities[id]:GetKeyValue("slideaxis") == "")) then
			-- Disallow the setting of an angle when hingeaxis and slide axis still exist as a key value!
			return false;
		end
	end
end

-- ############## And for _EntFire (void) @ aVoN
function VMFCG.core._EntFire_new(id,input,value,delay)
	if(not _EntExists(id)) then return nil end;
	-- This checks, if an entity, whichs standard alpha value is 255, has been added before (with maybe a lower alpha value) or not. Otherwise, don't add it to the VMFCG.Entities table (workaround for my PropProtector, which ent_fires props with alpha 255 on creation)
	-- Your question, why i do this? Smaller VMFCG.Entities table = better performance :D
	if(not VMFCG.Entities.Exists(id) and input == "alpha" and value == "255") then return end;
	VMFCG.Entities[id]:AddConnection(input,value);
end

-- ############## And for _phys.EnableMotion (void) @ aVoN
function VMFCG.core._PhysEnableMotion_new(id,motion)
	if(not _EntExists(id)) then return nil end;
	if(_EntGetType(id) == "prop_vehicle_jeep") then -- Prevents the game from crashing -> You cant freeze a jeep
		return false;
	end
	if(motion) then
		if(VMFCG.Entities.Exists(id)) then
			VMFCG.Entities[id]:SetKeyValue("_spawnfrozen",nil);
		end
	else
		VMFCG.Entities[id]:SetKeyValue("_spawnfrozen","true");
	end
end

-- ############## Overwrites the original _EntCreate() function with this one. It will call the ent kv/reset engine (void) @aVoN
function VMFCG.core._EntSpawn_new(id)
	local model = string.lower(_EntGetModel(id));
	for k,v in VMFCG.Entities do
		if(id ~= k and type(k) == "number") then
			if(v:GetModel() == model and model ~= "") then
				-- Same model - Now check the positions
				if(vecLength(vecSub(v:GetPos(),_EntGetPos(id))) <= 0.01) then
					-- Apply old key/values to the new entity
					VMFCG.AddTimerOnce("VMFCG.core.ApplyDeadKeyValues:"..k..":"..id,0.05,1,VMFCG.core.ApplyDeadKeyValues,id,v);
				end
			end
		end
	end
end

-- ############## This function is neccessary to save even phys_constraint which got welded with this way (void) @aVoN
function VMFCG.core._PhysConstraintSetEnts_new(id,attach1,attach2)
	if(not _EntExists(id)) then return nil end;
	VMFCG.Entities[id]:SetKeyValue("attach1",attach1);
	VMFCG.Entities[id]:SetKeyValue("attach2",attach2);
	VMFCG.debug("VMFCG.core._PhysConstraintSetEnts_new","Welded entity "..tostring(attach1).." and "..tostring(attach2).." on phys_constraint "..id);
end

-- ############## Hooks to _PhysEnableGravity (void) @ aVoN
function VMFCG.core._PhysEnableGravity_new(id,bool)
	if(not _EntExists(id)) then return nil end;
	if(bool) then
		VMFCG.Entities[id]:SetKeyValue("_gravity",nil);
	else
		VMFCG.Entities[id]:SetKeyValue("_gravity",false);
	end
end

-- ############## Hooks to _PhysEnableCollisions (void) @ aVoN
function VMFCG.core._PhysEnableCollisions_new(id,bool)
	if(not _EntExists(id)) then return nil end;
	if(bool) then
		VMFCG.Entities[id]:SetKeyValue("_collision",nil);
	else
		VMFCG.Entities[id]:SetKeyValue("_collision",false);
	end
end

-- ############## Hooks to _EntSetName (void) @ aVoN
function VMFCG.core._EntSetName_new(id,name)
	if(not _EntExists(id)) then return nil end;
	VMFCG.Entities.ChangeTargetName(id,name);
	VMFCG.Entities[id]:SetKeyValue("targetname",name);
end

--[[
-- Not neccessary anymore - Only added for testing purpose due to Entity Class changes
-- ############## Hooks to _EntSetParent (void) @ aVoN
function VMFCG.core._EntSetParent_new(id,parent)
	if(not _EntExists(id)) then return nil end;
	VMFCG.Entities[id]:SetParent(parent);
end
--]]

-- ############## It will automaticly apply old keyvalues of an prop to this new one, when the old got delted, and respawned on the same position, but with maybe another classname (like e.g. for dynafreezing) @aVoN
function VMFCG.core.ApplyDeadKeyValues(new_ent,dead_ent)
	-- Either, the old entity does not exist anymore, or the newly one spawned has the same id
	local id = dead_ent:GetId();
	if(not _EntExists(id) or new_ent == id) then
		local class = dead_ent:GetClass();
		local all_key_values = VMFCG.copy(dead_ent:GetAllKeyValues());
		local new_ent_obj = VMFCG.Ent:GetById(new_ent);
		local new_ent_kv = new_ent_obj:GetAllKeyValues();
		local dont_apply = {classname=true,_movetype=true,origin=true,angles=true,solid=true,spawnflags=true};
		for k,v in all_key_values do
			if(not dont_apply[k]) then
				new_ent_kv[k] = v;
			end
		end
		VMFCG.Entities[new_ent]:SetAllKeyValues(new_ent_kv);
		VMFCG.Entities[new_ent]:ApplyKeyValues();
		VMFCG.debug("VMFCG.core.ApplyDeadKeyValues","Copied key/values of killed entity ("..class..") "..id.." to (".._EntGetType(new_ent)..") "..new_ent.." because of model-match");
	end
end

-- #############################################
-- ############### Load,reload and kill part ################
-- #############################################


-- ############## Kills the VMFCG object and unbinds all hooks/thinks and timers @aVoN
function VMFCG.core.Kill()
	VMFCG.debug("VMFCG.core.Kill","Killing VMFCG Object");
	for k,v in VMFCG.Entities do
		if(type(k) == "number") then
			VMFCG.Entities.KillEnt(0,k,true);
		end
	end
	for k,v in VMFCG.hooks do
		for kk,vv in v do
			if(g_EvenHooks[vv] ~= nil) then
				UnHookEvent( vv ); 
				VMFCG.debug("VMFCG.Kill","Hook ("..k..") "..kk.." killed");
			end
			VMFCG.hooks[k][kk] = nil;
		end
	end
	for k,v in VMFCG.timers_once do
		if(Timer[v] ~= nil) then
			HaltTimer(v);
			VMFCG.timers_once[k] = nil;
			VMFCG.debug("VMFCG.Kill","Timer "..k.." killed");
		end
		VMFCG.timers_once[k] = nil;
	end
	for k,v in VMFCG.think_funcs do
		if(gLuaThinkFunctions[v] ~= nil) then
			gLuaThinkFunctions[v] = nil;
			VMFCG.debug("VMFCG.Kill","Think "..k.." killed");
		end
		VMFCG.think_funcs[v] = nil;
	end
	for k,v in VMFCG.selection.users do
		-- Hide huds!
		v.hud:Hide();
		-- And kill open-sprites
		VMFCG.selection.SphereSprite(k,false);
		-- And all open quads
		VMFCG.selection.QuadKill(k)
	end
	-- Reapply the key/value save engine
	VMFCG.core.overrideFunctionRemoveAll();
	VMFCG = nil;
end

-- ############## Reloads the script (void) @aVoN
function VMFCG.core.reload(userid,arg)
	if(VMFCG.RightsManager.hasAccess(userid,VMFCG.defines.ALLOW_SCRIPT_RELOAD)) then
		VMFCG.Hud.Message(userid,"                                         VMFCopyGun Reloaded!");
		_OpenScript("VMFCopyGun/vmfcg_init.lua");
		VMFCG.debug("VMFCG.reload","Restarted VMFCopyGun");
	end
end

-- ############## Kills the script first (removes it completely from the game) and reloads it (void) @aVoN
function VMFCG.core.kill_reload(userid,arg)
	if(VMFCG.RightsManager.hasAccess(userid,VMFCG.defines.ALLOW_SCRIPT_KILL_RELOAD)) then
		VMFCG.Hud.Message(userid,"                                       VMFCopyGun Kill-Reloaded!");
		VMFCG.core.Kill();
		_OpenScript("VMFCopyGun/vmfcg_init.lua");
		VMFCG.debug("VMFCG.kill_reload","Killed and restarted VMFCopyGun");
	end
end

-- #############################################
-- ############### PropProtector Wrapper ################
-- #############################################


-- ############## Checks, if a player is allowed to shoot or not (handles by PropProtector) (bool) @aVoN
function VMFCG.PropProtector_AllowedToShoot(userid)
	if(propprotector_version) then
		local steamid = _PlayerInfoCompatibility(userid,"networkid");
		return playerCanShoot[steamid]; -- Can he shoot?
	end
	return true;
end

-- ############## Checks, if a player is allowed to use an entity (handles by PropProtector) (bool) @aVoN
function VMFCG.PropProtector_AllowedToUseEntity(userid,ent)
	if(propprotector_version) then
		return prop_protection (userid,ent); -- Can he use the entity?
	end
	return true;
end

-- #############################################
-- ############### Misc Functions #####################
-- #############################################


-- ############## Handles the splashscreen on serverstartup (void) @aVoN
function VMFCG.SplashScreen()
	if(_file.Exists("lua/VMFCopyGun/data/vmfcg_splashscreen.dat")) then
		local sp = "\n".._file.Read("lua/VMFCopyGun/data/vmfcg_splashscreen.dat").."\n\n";
		_Msg(string.gsub(sp,"<version>",VMFCG.version));
	end
end


-- #############################################
-- ################### Init #######################
-- #############################################

-- ############## Override Functions
-- Key/Value set?
VMFCG.core.overrideFunctionAdd("_EntSetKeyValue","VMFCG.core._EntSetKeyValue_new");
-- New Material applyed?
VMFCG.core.overrideFunctionAdd("_EntSetMaterial","VMFCG.core._EntSetMaterial_new");
-- Outputs fired?
VMFCG.core.overrideFunctionAdd("_EntFire","VMFCG.core._EntFire_new");
-- Entity frozen?
VMFCG.core.overrideFunctionAdd("_phys.EnableMotion","VMFCG.core._PhysEnableMotion_new");
VMFCG.core.overrideFunctionAdd("_PhysEnableMotion","VMFCG.core._PhysEnableMotion_new");
-- Set attach1 and attach2 for an entity by this function?
VMFCG.core.overrideFunctionAdd("_phys.ConstraintSetEnts","VMFCG.core._PhysConstraintSetEnts_new");
VMFCG.core.overrideFunctionAdd("_PhysConstraintSetEnts","VMFCG.core._PhysConstraintSetEnts_new");
-- Gravity for the entitys changed?
VMFCG.core.overrideFunctionAdd("_phys.EnableGravity","VMFCG.core._PhysEnableGravity_new");
VMFCG.core.overrideFunctionAdd("_PhysEnableGravity","VMFCG.core._PhysEnableGravity_new");
-- Changes the collision for the VMFCG.Entities cache
VMFCG.core.overrideFunctionAdd("_phys.EnableCollisions","VMFCG.core._PhysEnableCollisions_new");
VMFCG.core.overrideFunctionAdd("_PhysEnableCollisions","VMFCG.core._PhysEnableCollisions_new");
-- Updates the VMFCG.Entities cache with new targetname, if changed by any scripts
VMFCG.core.overrideFunctionAdd("_EntSetName","VMFCG.core._EntSetName_new");
-- Handles, whether to allow to set an angle to an entity or not (when hinge/slideaxis are NOT set)
VMFCG.core.overrideFunctionAdd("_EntSetAng","VMFCG.core._EntSetAng_and_EntSetAngAngle_new");
VMFCG.core.overrideFunctionAdd("_EntSetAngAngle","VMFCG.core._EntSetAng_and_EntSetAngAngle_new");
-- Setting a model - model prechached? If not, lets do this by VMFCopyGun
VMFCG.core.overrideFunctionAdd("_EntSetModel","VMFCG.core._EntSetModel_new");
-- Ent set parent hook (not neccessary anymore - only for testing purpose due to changes in Entity Class)
--VMFCG.core.overrideFunctionAdd("_EntSetParent","VMFCG.core._EntSetParent_new");

-- ############## Normal init

--VMFCG.core.overrideFunctionAdd("_EntSpawn","VMFCG.core._EntSpawn_new"); -- Disabled, because it's currently very buggy!
VMFCG.core.getMaterialNames(); -- Fetch materials (core)

-- #############################################
-- ###############CONCOMMAND ####################
-- #############################################

VMFCG.Commands.Add("vmfcg_debug",VMFCG.show_debug);
VMFCG.Commands.Add("vmfcg_reload",VMFCG.core.reload);
VMFCG.Commands.Add("vmfcg_kreload",VMFCG.core.kill_reload);